-- ****************************************************************
-- Installs the shared InstantASP database tables and stored procedured
-- These are required by all InstantASP software
-- ****************************************************************

/******************************************************
  Tables  Begin
******************************************************/
GO
----------------------------------------------------
-- InstantASP_Administrators
----------------------------------------------------
CREATE TABLE InstantASP_Administrators
(
AdministratorID                     INT IDENTITY(1,1) NOT NULL,
UserID                              INT NOT NULL,
Notes                               NTEXT
)

GO
----------------------------------------------------
-- InstantASP_Emails
----------------------------------------------------
CREATE TABLE InstantASP_Emails
(
EmailID                             INT IDENTITY(1,1) NOT NULL,
EmailTo                             NVARCHAR(255) DEFAULT ('') NOT NULL,
EmailCC                             NVARCHAR(255) DEFAULT ('') NOT NULL,
EmailBCC                            NTEXT,
EmailFrom                           NVARCHAR(255) DEFAULT ('') NOT NULL,
EmailSubject                        NVARCHAR(1500) DEFAULT ('') NOT NULL,
EmailBody                           NTEXT,
EmailPriority                       TINYINT DEFAULT (1) NOT NULL,
EmailAttempts 			    TINYINT DEFAULT (0) NOT NULL,
DateStamp                           DATETIME DEFAULT (getdate()) NOT NULL,
CONSTRAINT PK_InstantASP_EmailQueue_EmailID PRIMARY KEY CLUSTERED ( EmailID )
)

GO
----------------------------------------------------
-- InstantASP_IgnoredSearchTerms
----------------------------------------------------
CREATE TABLE InstantASP_IgnoredSearchTerms
(
IgnoredSearchTermID                 INT IDENTITY(1,1) NOT NULL,
SearchText                          NVARCHAR(255) NOT NULL,
CONSTRAINT PK_InstantForum_SearchIgnoredTerms_IgnoredSearchTermID PRIMARY KEY CLUSTERED ( IgnoredSearchTermID )
)

GO
----------------------------------------------------
-- InstantASP_Roles
----------------------------------------------------
CREATE TABLE InstantASP_Roles
(
RoleID                              SMALLINT IDENTITY(1,1) NOT NULL,
RoleName                            NVARCHAR(200) DEFAULT (N'') NOT NULL,
RoleVisible                         BIT DEFAULT (0) NOT NULL,
OpenMarkup                          NVARCHAR(255) DEFAULT ('') NOT NULL,
CloseMarkup                         NVARCHAR(255) DEFAULT ('') NOT NULL,
AnonymousRole                       BIT DEFAULT (0) NOT NULL,
ActivationRole                      BIT DEFAULT (0) NOT NULL,
MemberRole                          BIT DEFAULT (0) NOT NULL,
ModeratorRole                       BIT DEFAULT (0) NOT NULL,
AdministratorRole                   BIT DEFAULT (0) NOT NULL,
BannedRole                          BIT DEFAULT (0) NOT NULL,
RequiredRole                        BIT DEFAULT (0) NOT NULL,
CONSTRAINT PK_InstantASP_Roles_RoleID PRIMARY KEY CLUSTERED ( RoleID )
)

GO
----------------------------------------------------
-- InstantASP_ScheduledTasks
----------------------------------------------------
CREATE TABLE InstantASP_ScheduledTasks
(
TaskID                              INT IDENTITY(1,1) NOT NULL,
TaskName                            NVARCHAR(255) DEFAULT ('') NOT NULL,
TaskDescription                     NVARCHAR(500) DEFAULT ('') NOT NULL,
TaskFileName                        NVARCHAR(255) DEFAULT ('') NOT NULL,
TaskNextRunDate                     DATETIME DEFAULT (getdate()) NOT NULL,
TaskLastRunDate                     DATETIME,
TaskInterval                        SMALLINT DEFAULT (0) NOT NULL,
TaskIntervalType                    TINYINT DEFAULT (1) NOT NULL,
TaskEnabled                         BIT DEFAULT (0) NOT NULL,
TaskGuid                            NVARCHAR(75) DEFAULT ('') NOT NULL,
DateStamp                           DATETIME DEFAULT (getdate()) NOT NULL,
CONSTRAINT PK_InstantForum_ScheduledTasks_TaskID PRIMARY KEY CLUSTERED ( TaskID )
)

GO
----------------------------------------------------
-- InstantASP_Sessions
----------------------------------------------------
CREATE TABLE InstantASP_Sessions
(
SessionID                           INT IDENTITY(1,1) NOT NULL,
SessionIdentity                     NVARCHAR(100) DEFAULT ('') NOT NULL,
SessionData                         NVARCHAR(1000) DEFAULT ('') NOT NULL,
Expires                             DATETIME DEFAULT (getdate()) NOT NULL,
CONSTRAINT PK_InstantASP_Sessions_SessionID PRIMARY KEY CLUSTERED ( SessionID )
)

GO
----------------------------------------------------
-- InstantASP_Settings
----------------------------------------------------
CREATE TABLE InstantASP_Settings
(
SettingID                           TINYINT IDENTITY(1,1) NOT NULL,
Culture                             NVARCHAR(10) DEFAULT (N'en-us') NOT NULL,
CookiePrefix                        NVARCHAR(25) DEFAULT ('IASP_') NOT NULL,
SMTPEmailEnabled                    BIT DEFAULT (1) NOT NULL,
SMTPServer                          NVARCHAR(255) DEFAULT (N'localhost') NOT NULL,
SMTPUsername                        NVARCHAR(255) DEFAULT (N'') NOT NULL,
SMTPPassword                        NVARCHAR(255) DEFAULT (N'') NOT NULL,
SMTPPort                            INT DEFAULT (25) NOT NULL,
RssEncoding                         NVARCHAR(25) DEFAULT (N'UTF-8') NOT NULL,
RssCacheMinutes                     INT DEFAULT (30) NOT NULL,
CompressOutput                      BIT DEFAULT (1) NOT NULL,
RemoveSpanTags                      BIT DEFAULT (1) NOT NULL,
TimeZoneOffset                      FLOAT DEFAULT (0) NOT NULL,
ObserveDaylightSavingTime           BIT DEFAULT (1) NOT NULL,
DateTimeFormat                      NVARCHAR(1) DEFAULT ('G') NOT NULL,
SMTPEmailType                       TINYINT DEFAULT (0) NOT NULL,
AuthenticationType                  TINYINT DEFAULT (1) NOT NULL,
AccountActivationType               TINYINT DEFAULT (1) NOT NULL,
DefaultUserRoleID                   SMALLINT DEFAULT (2) NOT NULL,
DefaultAwaitingApprovalRoleID       SMALLINT DEFAULT (6) NOT NULL,
AnonymousUsername                   NVARCHAR(255) DEFAULT (N'Anonymous') NOT NULL,
AnonymousEmailAddress               NVARCHAR(255) DEFAULT (N'someemail@yourdomain.com') NOT NULL,
RequireConfirmationOnEmailUpdate    BIT DEFAULT (1) NOT NULL,
FirstDayOfWeek                      TINYINT DEFAULT (0) NOT NULL,
LastLoginRefreshInterval            SMALLINT DEFAULT (20) NOT NULL,
HttpHeadersNoCache                  BIT DEFAULT (1) NOT NULL,
LoginUsing                          TINYINT DEFAULT (1) NOT NULL,
EmailsPerCycle                      TINYINT DEFAULT (50) NOT NULL,
EmailAttempts 			    TINYINT DEFAULT (3) NOT NULL,
SearchType                          TINYINT DEFAULT (1) NOT NULL,
CONSTRAINT PK_InstantASP_Settings_SettingID PRIMARY KEY CLUSTERED ( SettingID )
)

GO
----------------------------------------------------
-- InstantASP_Users
----------------------------------------------------
CREATE TABLE InstantASP_Users
(
UserID                              INT IDENTITY(1,1) NOT NULL,
EmailAddress                        NVARCHAR(255) DEFAULT ('') NOT NULL,
Password                            NVARCHAR(255) DEFAULT (N'') NOT NULL,
Username                            NVARCHAR(255) DEFAULT (N'') NOT NULL,
PrimaryRoleID                       SMALLINT DEFAULT (1) NOT NULL,
Culture                             NVARCHAR(20) DEFAULT (N'en-us') NOT NULL,
IPAddress                           NVARCHAR(20) DEFAULT (N'') NOT NULL,
TimeZoneOffset                      FLOAT DEFAULT (0) NOT NULL,
ObserveDaylightSavingTime           BIT DEFAULT (0) NOT NULL,
DateTimeFormat                      NVARCHAR(1) DEFAULT ('g') NOT NULL,
TotalVisits                         INT DEFAULT (1) NOT NULL,
CreatedDate                         DATETIME DEFAULT (getdate()) NOT NULL,
LastLoginDate                       DATETIME DEFAULT (getdate()) NOT NULL,
ConfirmationCode                    NVARCHAR(25) DEFAULT ('') NOT NULL,
FirstDayOfWeek                      TINYINT DEFAULT (0) NOT NULL,
CONSTRAINT PK_InstantASP_Users_UserID PRIMARY KEY CLUSTERED ( UserID )
)

GO
----------------------------------------------------
-- InstantASP_UsersRoles
----------------------------------------------------
CREATE TABLE InstantASP_UsersRoles
(
UserRoleID                          INT IDENTITY(1,1) NOT NULL,
UserID                              INT DEFAULT (0) NOT NULL,
RoleID                              SMALLINT DEFAULT (0) NOT NULL
)

GO
----------------------------------------------------
-- InstantASP_Wrappers
----------------------------------------------------
CREATE TABLE InstantASP_Wrappers
(
WrapperID                           SMALLINT IDENTITY(1,1) NOT NULL,
WrapperName                         NVARCHAR(255) DEFAULT (N'') NOT NULL,
HeaderHTML                          NTEXT,
FooterHTML                          NTEXT,
CONSTRAINT PK_InstantASP_Wrappers_WrapperID PRIMARY KEY CLUSTERED ( WrapperID )
)

GO

/******************************************************
  Tables  End
******************************************************/
GO
/******************************************************
  Foreign keys  Begin
******************************************************/
GO
/*-------- Foreign Keys-----------------*/
ALTER TABLE InstantASP_Administrators
    ADD CONSTRAINT FK_InstantASP_Admins_InstantASP_Users FOREIGN KEY( UserID )
    REFERENCES [InstantASP_Users] ( UserID )
    ON UPDATE NO ACTION
    ON DELETE CASCADE;

GO
/*-------- Foreign Keys-----------------*/
ALTER TABLE InstantASP_UsersRoles
    ADD CONSTRAINT FK_InstantASP_UserRoles_InstantASP_Users FOREIGN KEY( UserID )
    REFERENCES [InstantASP_Users] ( UserID )
    ON UPDATE NO ACTION
    ON DELETE CASCADE;
ALTER TABLE InstantASP_UsersRoles
    ADD CONSTRAINT FK_InstantASP_UsersRoles_InstantASP_Roles FOREIGN KEY( RoleID )
    REFERENCES [InstantASP_Roles] ( RoleID )
    ON UPDATE NO ACTION
    ON DELETE CASCADE;

GO

/******************************************************
  Foreign keys  End
******************************************************/
GO
/******************************************************
  Stored procedures  Begin
******************************************************/
GO
----------------------------------------------------
-- iasp_sp_ActivateUserAccount
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO


CREATE PROCEDURE iasp_sp_ActivateUserAccount (
@intUserID int,
@strKey nvarchar(25),
@intRoleID int,
@bitFound bit output
) AS

SET NOCOUNT ON 

/* If a matching confirmation key and user identity are found this proc moves the user to the role specified in the @intRoleID input parameter */

IF EXISTS(SELECT UserID FROM InstantASP_Users WHERE (UserID = @intUserID AND ConfirmationCode = @strKey))
BEGIN /* We have a match */

	-- Reset permission ID to zero to ensure the application uses the permissions set associated with the primary role identity 
	-- Also update the users primary role association and remove the confirmation code as they are now confirmed
	UPDATE InstantASP_Users SET PrimaryRoleID = @intRoleID, ConfirmationCode ='' WHERE  (UserID = @intUserID)

	-- Indicate this was a success 
	SET @bitFound = 1
END
ELSE
BEGIN
	SET @bitFound = 0
END

RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_DeleteAllQueuedEmails
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_DeleteAllQueuedEmails
 AS

SET NOCOUNT ON 

/* Deletes all queued email within InstantASP_Emails */

DELETE FROM InstantASP_Emails
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_DeleteIgnoredSearchTerm
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_DeleteIgnoredSearchTerm (
@intIgnoredSearchTermID int
)
AS

SET NOCOUNT ON 

/* Deletes an ignored search term from the InstantASP_IgnoredSearchTerms table table */

DELETE FROM InstantASP_IgnoredSearchTerms WHERE (InstantASP_IgnoredSearchTerms.IgnoredSearchTermID = @intIgnoredSearchTermID)

RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_DeleteQueuedEmails
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_DeleteQueuedEmails (
@strIdentities nvarchar(2000)
) AS

SET NOCOUNT ON 

/* Accepts a comma delimited list of email identities and loops through this list removing emails matching the identities */

DECLARE @SQL nvarchar(1500)
DECLARE @Separator varchar(1)
DECLARE @Value varchar(255)
DECLARE @SepPosition int
DECLARE @LoopCount int

SET  @Separator = ","
SET @LoopCount = 0
SET @strIdentities = @strIdentities + @Separator

WHILE patindex('%' + @Separator + '%' , @strIdentities) <> 0 -- loop until we find no more separators

	BEGIN
	
		SET @SepPosition =  patindex('%' + @Separator + '%' , @strIdentities)  -- patindex matches the a pattern against a string
		SET @Value = left(@strIdentities, @SepPosition - 1) -- array value holds each single keyword
		
		/* Delete email as this has now been successfully sent */	
		 DELETE FROM InstantASP_Emails WHERE EmailID = @Value  
		
		SET @LoopCount = + 1 -- increment loop count		
		SET @strIdentities = stuff(@strIdentities, 1, @SepPosition, '') -- This replaces what we just processed with and empty string, pevents infinate loops
	END
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_DeleteRole
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO

CREATE PROCEDURE iasp_sp_DeleteRole (
@intRoleID smallint,
@intMoveToRoleID smallint
) AS
SET NOCOUNT ON 

/* Deletes a role from the InstantASP_Roles table and moves any users belonging to that role to a new role */

-- update users primary role within InstantASP_Users to new role
IF (@intMoveToRoleID > 0)
BEGIN
	-- update users primary role identity
	UPDATE InstantASP_Users SET InstantASP_Users.PrimaryRoleID = @intMoveToRoleID WHERE (InstantASP_Users.PrimaryRoleID = @intRoleID)
END

-- add new role relationship to InstantASP_UsersRoles for each user within the old role we are deleting
DECLARE @intUserID int
DECLARE MSGCURSOR CURSOR FOR SELECT InstantASP_Users.UserID FROM InstantASP_Users
(nolock) WHERE (InstantASP_Users.PrimaryRoleID = @intMoveToRoleID)
	
OPEN MSGCURSOR
	
FETCH NEXT FROM MSGCURSOR
INTO @intUserID
	
WHILE @@FETCH_STATUS = 0
BEGIN
	
	-- check to ensure the user / role relationship does not already exist
	IF NOT EXISTS(SELECT UserID FROM InstantASP_UsersRoles WHERE InstantASP_UsersRoles.UserID = @intUserID AND InstantASP_UsersRoles.RoleID = @intMoveToRoleID)
	BEGIN
		-- add the user / role relationship
		INSERT INTO InstantASP_UsersRoles (UserID, RoleID) VALUES (@intUserID, @intMoveToRoleID)
	END

	-- fetch next user within old role
	FETCH NEXT FROM MSGCURSOR
	INTO @intUserID
	
END

-- delete the role from the InstantASP_Roles table, this also cascades deletes to InstantASP_UsersRoles for the old role
DELETE FROM InstantASP_Roles WHERE (InstantASP_Roles.RoleID = @intRoleID AND InstantASP_Roles.RequiredRole = 0)

RETURN
GO

SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_DeleteScheduledTask
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_DeleteScheduledTask (
@intTaskID int
)
AS

SET NOCOUNT ON 

/* Deletes a task from the InstantASP_ScheduledTasks table */

DELETE FROM InstantASP_ScheduledTasks WHERE (InstantASP_ScheduledTasks.TaskID = @intTaskID)

RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_DeleteSessionData
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_DeleteSessionData (
@strSessionIdentity nvarchar(200)
)
AS

SET NOCOUNT ON 

/* Removes session data from the sessions table for a specific session identity */

DELETE FROM InstantASP_Sessions WHERE (InstantASP_Sessions.SessionIdentity = @strSessionIdentity)

RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_DeleteUser
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_DeleteUser (
@intUserID int
)
AS

SET NOCOUNT ON 

/* Deletes a user from the InstantASP_Users table, role deletes are cascaded */

DELETE FROM InstantASP_Users WHERE (InstantASP_Users.UserID = @intUserID)

RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_DeleteUserRole
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_DeleteUserRole (
@intUserID int,
@intRoleID smallint
) AS

SET NOCOUNT ON 

/* Deletes a user & role association from the InstantASP_UsersRoles table */

DELETE FROM InstantASP_UsersRoles  WHERE (UserID = @intUserID AND RoleID = @intRoleID)

RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_DeleteUserRolesRelationships
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO


CREATE PROCEDURE iasp_sp_DeleteUserRolesRelationships (
@intUserID int
)
AS

SET NOCOUNT ON 

/* Deletes all roles associated to a specific user from the InstantASP_UsersRoles table */

DELETE FROM InstantASP_UsersRoles WHERE (InstantASP_UsersRoles.UserID = @intUserID)
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_DeleteWrapper
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO


CREATE PROCEDURE iasp_sp_DeleteWrapper (
@intWrapperID smallint
)
AS

SET NOCOUNT ON 

/* Delets a wrapper from the InstantASP_Wrappers table */

DELETE FROM InstantASP_Wrappers WHERE (InstantASP_Wrappers.WrapperID = @intWrapperID)

RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_ExecuteSQL
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_ExecuteSQL (
@strSQL ntext
)
AS

SET NOCOUNT ON 

/* Executes any dynamic SQL, dynamic SQL is sometimes generated in code as opposed to stored procedures for performance reasons, 
    for example cursors are bad we would rather reiterate through records in code also building complex sql is preferred in code as SQL Server concateination is pants */

EXECUTE sp_executesql @strSQL
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_ExpireSessionData
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO


CREATE PROCEDURE iasp_sp_ExpireSessionData (
@strDateTime nvarchar(255)
)
AS

SET NOCOUNT ON 

/* Removes expired data from the sessions table */

DELETE FROM InstantASP_Sessions WHERE (InstantASP_Sessions.Expires <= @strDateTime) -- tidy session data 

RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_InsertQueuedEmail
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO


CREATE PROCEDURE iasp_sp_InsertQueuedEmail (
@strEmailTo nvarchar(255),
@strEmailCC nvarchar(255),
@strEmailBCC ntext,
@strEmailFrom nvarchar(255),
@strEmailSubject nvarchar(1500),
@strEmailBody ntext,
@intEmailPriority tinyint
) AS

SET NOCOUNT ON 

/* Inserts a new email into the InstantASP_Emails mail queue */

INSERT INTO InstantASP_Emails(EmailTo, EmailCC, EmailBCC, EmailFrom, EmailSubject, EmailBody, EmailPriority, DateStamp) VALUES
(@strEmailTo, @strEmailCC, @strEmailBCC, @strEmailFrom, @strEmailSubject, @strEmailBody, @intEmailPriority, GetDate())
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_InsertSessionData
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_InsertSessionData (
@strSessionID nvarchar(100),
@strSessionData nvarchar(1000),
@intExpires int
)
AS

DECLARE @DataKey nvarchar(255)
SET @DataKey = SUBSTRING(@strSessionData,1,CHARINDEX(":",@strSessionData)-1)

/* Adds new information that needs to be persistent throughout current session */

SET NOCOUNT ON 
	
	DELETE FROM InstantASP_Sessions WHERE (SessionIdentity = @strSessionID AND SessionData LIKE @DataKey + ':%') -- remove any existing session data
	INSERT INTO InstantASP_Sessions (SessionIdentity, SessionData, Expires) VALUES (@strSessionID, @strSessionData, DateAdd(n, @intExpires,GetDate())) -- add new session data

RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_InsertUpdateIgnoredSearchTerm
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO




CREATE PROCEDURE iasp_sp_InsertUpdateIgnoredSearchTerm (
@intIgnoredSearchTermID int,
@strSearchText nvarchar(255),
@intIdentity int output
) AS

SET NOCOUNT ON 

/* Inserts or updates an ignored search terms within the InstantASP_IgnoredSearchTerms table */

-- is this an update or insert?
IF (@intIgnoredSearchTermID = 0)
BEGIN
	
	IF NOT EXISTS(SELECT IgnoredSearchTermID FROM InstantASP_IgnoredSearchTerms WHERE SearchText = @strSearchText)
	BEGIN
		-- insert the term
		INSERT INTO InstantASP_IgnoredSearchTerms (SearchText) VALUES
		(@strSearchText)
	
		-- return the terms new identity
		SET @intIdentity = @@IDENTITY
		RETURN
	END

END
ELSE -- we are updating an existing term
BEGIN
	-- update the wrapper
	UPDATE InstantASP_IgnoredSearchTerms SET
	SearchText = @strSearchText
	WHERE InstantASP_IgnoredSearchTerms.IgnoredSearchTermID = @intIgnoredSearchTermID

	-- return the role identity
	SET @intIdentity = @intIgnoredSearchTermID
	RETURN
END

SET @intIdentity = 0
RETURN
GO

SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_InsertUpdateRole
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_InsertUpdateRole (
@intRoleID int,
@strRoleName nvarchar(200),
@bitRoleVisible bit,
@strOpenMarkUp nvarchar(255),
@strCloseMarkUp nvarchar(255),
@intIdentity int output
) AS

SET NOCOUNT ON 

/* Inserts or updates a user role within the InstantASP_Roles table */

-- is this an update or insert?
IF (@intRoleID = 0)
BEGIN
	-- insert the role
	INSERT INTO InstantASP_Roles (RoleName, RoleVisible, OpenMarkUp, CloseMarkUp) VALUES
	(@strRoleName, @bitRoleVisible, @strOpenMarkUp, @strCloseMarkUp)

	-- return the roles new identity
	SET @intIdentity = @@IDENTITY
END
ELSE -- we are updating an existing role
BEGIN
	-- update the role
	UPDATE InstantASP_Roles SET
	RoleName = @strRoleName,
	RoleVisible = @bitRoleVisible,
	OpenMarkUp = @strOpenMarkUp,
	CloseMarkUp = @strCloseMarkUp 
	WHERE InstantASP_Roles.RoleID = @intRoleID

	-- return the role identity
	SET @intIdentity = @intRoleID
END

RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_InsertUpdateScheduledTask
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_InsertUpdateScheduledTask (
@intTaskID int,
@strTaskName nvarchar(255),
@strTaskDescription nvarchar(500),
@strTaskFileName nvarchar(255),
@strTaskNextRunDate nvarchar(255),
@intTaskInterval int,
@intTaskIntervalType tinyint,
@bitTaskEnabled bit,
@intIdentity int output
) AS

SET NOCOUNT ON 

/* Inserts or updates a scheduled task within the InstantASP_ScheduledTasks table */

-- is this an update or insert?
IF (@intTaskID = 0)
BEGIN
	-- insert the role
	INSERT INTO InstantASP_ScheduledTasks (TaskName, TaskDescription, TaskFileName, TaskNextRunDate, TaskInterval, TaskIntervalType, TaskEnabled, DateStamp) VALUES
	(@strTaskName, @strTaskDescription, @strTaskFileName, @strTaskNextRunDate, @intTaskInterval, @intTaskIntervalType, @bitTaskEnabled, GetDate())

	-- return the tasks new identity
	SET @intIdentity = @@IDENTITY
	RETURN

END
ELSE -- we are updating an existing task
BEGIN
	-- update the task
	UPDATE InstantASP_ScheduledTasks SET
	TaskName = @strTaskName,
	TaskDescription = @strTaskDescription,
	TaskFileName = @strTaskFileName,
	TaskNextRunDate = @strTaskNextRunDate,
	TaskInterval = @intTaskInterval,
	TaskIntervalType = @intTaskIntervalType,
	TaskEnabled = @bitTaskEnabled
	WHERE InstantASP_ScheduledTasks.TaskID = @intTaskID

	-- return the task identity
	SET @intIdentity = @intTaskID
	RETURN
END

SET @intIdentity = 0
RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_InsertUpdateWrapper
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_InsertUpdateWrapper (
@intWrapperID int,
@strWrapperName nvarchar(255),
@strWrapperHeader ntext,
@strWrapperFooter ntext,
@intIdentity int output
) AS

SET NOCOUNT ON 

/* Inserts or updates a HTML wrapper within the InstantASP_Roles table */

-- is this an update or insert?
IF (@intWrapperID = 0)
BEGIN
	-- insert the role
	INSERT INTO InstantASP_Wrappers (WrapperName, HeaderHTML, FooterHTML) VALUES
	(@strWrapperName, @strWrapperHeader, @strWrapperFooter)

	-- return the wrappers new identity
	SET @intIdentity = @@IDENTITY
	RETURN

END
ELSE -- we are updating an existing wrapper
BEGIN
	-- update the wrapper
	UPDATE InstantASP_Wrappers SET
	WrapperName = @strWrapperName,
	HeaderHTML = @strWrapperHeader,
	FooterHTML = @strWrapperFooter
	WHERE InstantASP_Wrappers.WrapperID = @intWrapperID

	-- return the role identity
	SET @intIdentity = @intWrapperID
	RETURN
END

SET @intIdentity = 0
RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_InsertUser
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO


CREATE PROCEDURE iasp_sp_InsertUser (
@strEmailAddress nvarchar(255),
@strPassword nvarchar(255),
@strUsername nvarchar(255),
@intPrimaryRoleID int,
@strCulture nvarchar(20),
@strIPAddress nvarchar(20),
@dblTimeZoneOffset float,
@bitObserveDaylightSavingTime bit,
@strDateTimeFormat char(1),
@strConfirmationCode nvarchar(25),
@intIdentity int output
) AS
SET NOCOUNT ON 
/* Ensure the email address is not already present within the table */
IF EXISTS(SELECT UserID FROM InstantASP_Users WHERE InstantASP_Users.EmailAddress = @strEmailAddress)
BEGIN 
	SET @intIdentity = 0
	RETURN
END
/* Ensure the username is not already present within the table */
IF EXISTS(SELECT UserID FROM InstantASP_Users WHERE InstantASP_Users.Username = @strUsername)
BEGIN 
	SET @intIdentity = -1
	RETURN
END
/* Inserts a new user into the InstantASP_Users table and returns the new user identity */
INSERT INTO InstantASP_Users (EmailAddress, Username, [Password], PrimaryRoleID, Culture, IPAddress, TimeZoneOffset, ObserveDaylightSavingTime, DateTimeFormat, TotalVisits, CreatedDate, LastLoginDate, ConfirmationCode)
VALUES (@strEmailAddress, @strUsername, @strPassword, @intPrimaryRoleID, @strCulture, @strIPAddress, @dblTimeZoneOffset, @bitObserveDaylightSavingTime, @strDateTimeFormat, 0, GetDate(), GetDate(), @strConfirmationCode)
SET @intIdentity = @@IDENTITY
RETURN
GO

SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_InsertUserRole
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_InsertUserRole (
@intUserID int,
@intRoleID smallint
) AS

SET NOCOUNT ON 

/* Inserts a new user & role association to the InstantASP_UsersRoles table */

-- check to ensure we don't already have the role association
IF NOT EXISTS(SELECT RoleID From InstantASP_UsersRoles WHERE UserID = @intUserID AND RoleID = @intRoleID)
BEGIN
	INSERT INTO InstantASP_UsersRoles (UserID, RoleID) VALUES (@intUserID, @intRoleID)
END

RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_PagingProcedure
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_PagingProcedure (
@strTables varchar(1500),
@strPK nvarchar(100),
@strSort nvarchar(300) = null,
@intPageNumber int = 1,
@intPageSize int = 10,
@strFields nvarchar(1500) = '*',
@strFilter nvarchar(1000) = null,
@strGroup nvarchar(1000) = null
)
AS

SET NOCOUNT ON 

/* This stored procedure is used to provide a quick way to return a specific range of rows based on the current page index. 
For example when using a datagrid to present data we don't have to return the entire resultset we simply return the rows required. */

/*Default Sorting*/

IF (@strSort IS null OR @strSort = '')
BEGIN
	SET @strSort = @strPK
END

/*Find the @PK type*/
DECLARE @SortTable nvarchar(100)
DECLARE @SortName nvarchar(100)
DECLARE @strSortColumn nvarchar(200)
DECLARE @operator char(2)
DECLARE @type nvarchar(100)
DECLARE @query nvarchar(2000)

/*Set sorting variables.*/	
IF CHARINDEX('DESC',@strSort)>0
	BEGIN
		SET @strSortColumn = REPLACE(@strSort, 'DESC', '')
		SET @operator = '<='
	END
ELSE
	BEGIN
		IF CHARINDEX('ASC', @strSort) = 0
			SET @strSortColumn = REPLACE(@strSort, 'ASC', '')
		SET @operator = '>='
	END


IF CHARINDEX('.', @strSortColumn) > 0
	BEGIN
		SET @SortTable = SUBSTRING(@strSortColumn, 0, CHARINDEX('.',@strSortColumn))
		SET @SortName = SUBSTRING(@strSortColumn, CHARINDEX('.',@strSortColumn) + 1, LEN(@strSortColumn))
	END
ELSE
	BEGIN
		SET @SortTable = @strTables
		SET @SortName = @strSortColumn
	END

DECLARE @prec int
SELECT @type = t.name, @prec = c.prec FROM sysobjects o JOIN syscolumns c on o.id = c.id JOIN systypes t on c.xusertype = t.xusertype WHERE o.name = @SortTable AND c.name = @SortName

IF CHARINDEX('char', @type) > 0
   SET @type = @type + '(' + CAST(@prec AS varchar) + ')'

-- Optional Code No Dependency on sysobjects however not as accurate, 
-- at least 1 known bug when sorting nvarchar datatypes
-- No noticeable performance gains measured with this code over the above
--IF COL_LENGTH(@SortTable, @SortName) = 8
--  SET @type = 'datetime'
--ELSE IF COL_LENGTH(@SortTable, @SortName) = 4
--  SET @type = 'int'
--ELSE IF COL_LENGTH(@SortTable, @SortName) = 1
--  SET @type = 'bit'
--ELSE IF COL_LENGTH(@SortTable, @SortName) > 8
--  SET @type = 'nvarchar'

DECLARE @strPageSize varchar(50)
DECLARE @strStartRow varchar(50)
DECLARE @strSimpleFilter varchar(1000)
DECLARE @strLocalFilter varchar(1000)
DECLARE @strLocalGroup varchar(1000)

/*Default Page Number*/
IF @intPageNumber < 1
	SET @intPageNumber = 1

/*Set paging variables.*/
SET @strPageSize = CAST(@intPageSize AS varchar(50))
SET @strStartRow = CAST(((@intPageNumber - 1) * @intPageSize + 1) AS varchar(50))

/*Set filter & group variables.*/
IF (@strFilter IS NOT NULL AND @strFilter != '')
	BEGIN
		SET @strLocalFilter = ' WHERE ' + @strFilter + ' '
		SET @strSimpleFilter = ' AND (' + @strFilter + ') '
	END
ELSE
	BEGIN
		SET @strSimpleFilter = ''
		SET @strLocalFilter = ''
	END
IF @strGroup IS NOT NULL AND @strGroup != ''
	SET @strLocalGroup = ' GROUP BY ' + @strGroup + ' '
ELSE
	SET @strLocalGroup = ''
	
/*Build dynamic query*/
DECLARE @SQL nvarchar(3000)
SELECT @SQL = '
DECLARE @SortColumn ' + @type + '
DECLARE @LastValue int
SET ROWCOUNT ' + @strStartRow + '
SELECT @SortColumn = isnull(' + @strSortColumn + ',''''), @LastValue = ' + @strPK + ' FROM ' + @strTables + @strLocalFilter + ' ' + @strLocalGroup + ' ORDER BY ' + @strSort + ', ' + @strPK + '
SET ROWCOUNT ' + @strPageSize + '
SELECT ' + @strFields + ' FROM ' + @strTables + ' WHERE (' + @strSortColumn + @operator + ' @SortColumn) AND (' + @strPK + ' <> @LastValue  OR @SortColumn >= ' + @strSortColumn + ')' + @strSimpleFilter + ' ' + @strLocalGroup + ' 
ORDER BY ' + @strSort +  ', ' + @strPK

/*Execute dynamic query*/
EXECUTE sp_executesql @SQL
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_RecordCount
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_RecordCount (
@strTables nvarchar(2000),
@strWhereClause nvarchar(2000),
@strField nvarchar(2000),
@intRecordCount int output
)
AS

SET NOCOUNT ON 

/* Returns a record count based on the sql passed into the procedure via the input parameters */

DECLARE @SQL nvarchar(3000), @Param nvarchar(1000), @rtnVal nvarchar(25), @WhereClause nvarchar(1000)

-- Do we have a where clause 
IF (@strWhereClause IS NOT NULL AND @strWhereClause != '')
	BEGIN
		SET @WhereClause = ' WHERE (' + @strWhereClause + ')'
	END
	ELSE
	BEGIN
		SET @WhereClause = ''
	END

SELECT @SQL = 'SELECT @rtnVal = count(' + @strField +') FROM ' + @strTables + @WhereClause
SELECT @Param = '@rtnVal nvarchar(1000) OUTPUT'
EXECUTE sp_executesql @SQL, @Param, @rtnVal OUTPUT 
SET @intRecordCount = (Select @rtnVal)
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectAdministrator
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_SelectAdministrator (
@intUserID int
) 
AS

SET NOCOUNT ON 

/* Returns all information for a specific administrator */

SELECT * FROM InstantASP_Administrators (nolock) WHERE (InstantASP_Administrators.UserID = @intUserID)
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectExpiredScheduledTasks
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO


CREATE PROCEDURE iasp_sp_SelectExpiredScheduledTasks (
@strDateTime nvarchar(255)
)
AS

SET NOCOUNT ON 

/* Select all tasks where the run at date has passed the current date ie tasks we need to run */

SELECT * FROM InstantASP_ScheduledTasks WHERE (InstantASP_ScheduledTasks.TaskNextRunDate <= @strDateTime) -- tidy session data 

RETURN
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectIgnoredSearchTerms
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_SelectIgnoredSearchTerms
AS

SET NOCOUNT ON 

/* Select all ignored search terms */
/* These are words that just typically add more noise to a search such as and, or, this, that, of, then etc */
/* Note how we sort via the clustered primary key for performance */

SELECT * FROM InstantASP_IgnoredSearchTerms (nolock)
ORDER BY InstantASP_IgnoredSearchTerms.IgnoredSearchTermID DESC
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectQueuedEmails
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO

CREATE PROCEDURE iasp_sp_SelectQueuedEmails (
@intResultLimit int
)
AS
SET NOCOUNT ON 
/* Returns emails within current queue ordered by priority */
DECLARE @strSQL nvarchar(500)
SELECT @strSQL = 'SELECT TOP ' + CAST(@intResultLimit as nvarchar(25)) + ' * FROM InstantASP_Emails (nolock) ORDER BY DateStamp DESC, EmailPriority ASC'
EXECUTE sp_executesql @strSQL
GO

SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectRole
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_SelectRole (
@strRoleName nvarchar(255)
)
AS

SET NOCOUNT ON 

/* Returns All Information for a specific Role */

SELECT * FROM InstantASP_Roles (nolock) WHERE RoleName = @strRoleName
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectRoleByID
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_SelectRoleByID (
@intRoleID int
)
AS

SET NOCOUNT ON 

/* Returns All Information for a specific Role */

SELECT * FROM InstantASP_Roles (nolock) WHERE RoleID = @intRoleID
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectRoles
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_SelectRoles 
AS

SET NOCOUNT ON 

/* Returns all roles */

SELECT * FROM InstantASP_Roles (nolock) ORDER BY RoleID ASC
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectScheduledTasks
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_SelectScheduledTasks 
AS

SET NOCOUNT ON 

/* Returns all scheduled tasks */

SELECT * FROM InstantASP_ScheduledTasks (nolock) ORDER BY InstantASP_ScheduledTasks.DateStamp DESC
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectSessionData
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_SelectSessionData (
@strSessionID nvarchar(255)
)
AS

SET NOCOUNT ON 

/* Returns all session information for a specific session identity */

SELECT SessionData FROM InstantASP_Sessions (nolock) WHERE SessionIdentity = @strSessionID
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectSettings
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_SelectSettings 
AS

SET NOCOUNT ON 

/* Selects application settings that can be shared across InstantASP applications */

SELECT * FROM InstantASP_Settings (nolock)
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectSQLStatistics
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO


CREATE PROCEDURE iasp_sp_SelectSQLStatistics
AS

SET NOCOUNT ON 

/* Selects important SQL Server diagnostics information*/

SELECT @@VERSION AS Version, @@LANGUAGE AS Language, @@CONNECTIONS AS Connections, @@IDLE AS IDLE, @@IO_BUSY AS IOBusy, 
@@PACK_RECEIVED AS PackReceived, @@PACK_SENT AS PackSent, @@PACKET_ERRORS AS PackErrors, @@TOTAL_READ AS TotalReads, @@TOTAL_WRITE AS TotalWrites,
@@TOTAL_ERRORS AS TotalErrors
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectStoredProcedures
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_SelectStoredProcedures
AS

SELECT Name FROM sysobjects WHERE
(xtype = 'p') AND (name NOT LIKE 'dt%')
ORDER BY name
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectTables
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO


CREATE Procedure iasp_sp_SelectTables
AS

SET NOCOUNT ON

CREATE TABLE #TBLSize (TableName varchar(100), TableRows int, TableReserved varchar(80), TableData varchar(80), TableIndexSize varchar(80), TableUnused varchar(80))

DECLARE @TableName varchar(80) 

DECLARE MSGCURSOR CURSOR FOR SELECT NAME FROM sysobjects WHERE xType = 'U'

OPEN MSGCURSOR

FETCH NEXT FROM MSGCURSOR 
INTO @TableName

	WHILE @@FETCH_STATUS = 0
	BEGIN
	
	   INSERT INTO #tblSize(TableName, TableRows, TableReserved, TableData, TableIndexSize, TableUnused)
	   EXEC Sp_SpaceUsed @TableName
	      
	   -- Get the next table.
	   FETCH NEXT FROM MSGCURSOR 
	   INTO @TableName
	END

CLOSE MSGCURSOR
DEALLOCATE MSGCURSOR

-- select on the instant tables
SELECT TableName, TableRows, TableReserved, TableData, TableIndexSize, TableUnused
FROM #tblSize WHERE (TableName LIKE 'Instant%')  ORDER BY TableName 

DROP TABLE #TblSize

return
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectUserCountForRoles
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_SelectUserCountForRoles (
@intRoleID int,
@bitPrimary bit,
@intCount int output
) AS

SET NOCOUNT ON 

/* Counts the primary or secondary users belonging to any role */

IF (@bitPrimary = 1)
BEGIN
	SET @intCount = (SELECT COUNT(InstantASP_Users.UserID) FROM InstantASP_Users (nolock) INNER JOIN
              InstantASP_UsersRoles ON InstantASP_Users.UserID = InstantASP_UsersRoles.UserID 
	WHERE (InstantASP_Users.PrimaryRoleID = @intRoleID) AND (InstantASP_UsersRoles.RoleID = @intRoleID))
END
ELSE
BEGIN
	SET @intCount = (SELECT COUNT(InstantASP_Users.UserID) FROM InstantASP_Users (nolock)  INNER JOIN
              InstantASP_UsersRoles ON InstantASP_Users.UserID = InstantASP_UsersRoles.UserID 
	WHERE (InstantASP_Users.PrimaryRoleID <> @intRoleID) AND (InstantASP_UsersRoles.RoleID = @intRoleID))
END
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectUserData
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO

CREATE PROCEDURE iasp_sp_SelectUserData (
@strUsername nvarchar(255),
@strPassword nvarchar(255),
@strEncyptedPassword nvarchar(255),
@intLoginUsing int,
@intUserID int
) AS
SET NOCOUNT ON 
DECLARE @RoleNames nvarchar(600)
/* Select user information from the main InstantASP_Users table */
/* @intLoginUsing determines if we should use the username or email address field for the login field
    @intLoginUsing = 1 we use the email address
    @intLoginUsing = 2 we use the username */
/* If we have a user identity use this to select data */
IF (@intUserID > 0)
BEGIN
		
		/* Build a pipe delimited list of associated role names */
	
		SELECT @RoleNames = COALESCE (@RoleNames + '|', '') + InstantASP_Roles.RoleName
		FROM InstantASP_Roles (nolock) LEFT OUTER JOIN InstantASP_UsersRoles ON InstantASP_Roles.RoleID = InstantASP_UsersRoles.RoleID LEFT OUTER JOIN
		InstantASP_Users ON InstantASP_UsersRoles.UserID = InstantASP_Users.UserID WHERE (InstantASP_Users.UserID = @intUserID)
		SELECT @RoleNames As Roles, InstantASP_Users.* FROM InstantASP_Users (nolock) WHERE (InstantASP_Users.UserID = @intUserID) 
END
ELSE
BEGIN
	
	-- check member based on email or username (see above) and password
	IF (@strPassword <> '' AND @strEncyptedPassword <> '')
	BEGIN
		-- login using email address
		IF  (@intLoginUsing = 1) 
		BEGIN
	
			/* Build a pipe delimited list of associated role names */
			SELECT @RoleNames = COALESCE (@RoleNames + '|', '') + InstantASP_Roles.RoleName
			FROM InstantASP_Roles  (nolock) LEFT OUTER JOIN InstantASP_UsersRoles ON InstantASP_Roles.RoleID = InstantASP_UsersRoles.RoleID LEFT OUTER JOIN
			InstantASP_Users ON InstantASP_UsersRoles.UserID = InstantASP_Users.UserID WHERE (InstantASP_Users.EmailAddress = @strUsername)  AND (InstantASP_Users.[Password] = @strPassword OR InstantASP_Users.[Password] = @strEncyptedPassword)
			SELECT @RoleNames As Roles, InstantASP_Users.* FROM InstantASP_Users (nolock) WHERE (InstantASP_Users.EmailAddress = @strUsername)  AND (InstantASP_Users.[Password] = @strPassword OR InstantASP_Users.[Password] = @strEncyptedPassword)
		END
		ELSE IF   (@intLoginUsing = 2) -- login using username
		BEGIN
			/* Build a pipe delimited list of associated role names */
			SELECT @RoleNames = COALESCE (@RoleNames + '|', '') + InstantASP_Roles.RoleName
			FROM InstantASP_Roles  (nolock) LEFT OUTER JOIN InstantASP_UsersRoles ON InstantASP_Roles.RoleID = InstantASP_UsersRoles.RoleID LEFT OUTER JOIN
			InstantASP_Users ON InstantASP_UsersRoles.UserID = InstantASP_Users.UserID WHERE (InstantASP_Users.Username = @strUsername)  AND (InstantASP_Users.[Password] = @strPassword OR InstantASP_Users.[Password] = @strEncyptedPassword)
			SELECT @RoleNames As Roles, InstantASP_Users.* FROM InstantASP_Users (nolock) WHERE (InstantASP_Users.Username = @strUsername)  AND (InstantASP_Users.[Password] = @strPassword OR InstantASP_Users.[Password] = @strEncyptedPassword)
		END
	END
	ELSE
	BEGIN
	
		/* Build a pipe delimited list of associated role names */
	
		SELECT @RoleNames = COALESCE (@RoleNames + '|', '') + InstantASP_Roles.RoleName
		FROM InstantASP_Roles  (nolock) LEFT OUTER JOIN InstantASP_UsersRoles ON InstantASP_Roles.RoleID = InstantASP_UsersRoles.RoleID LEFT OUTER JOIN
		InstantASP_Users ON InstantASP_UsersRoles.UserID = InstantASP_Users.UserID WHERE (InstantASP_Users.EmailAddress = @strUsername OR InstantASP_Users.Username = @strUsername)
		SELECT @RoleNames As Roles, InstantASP_Users.* FROM InstantASP_Users (nolock) WHERE (InstantASP_Users.EmailAddress = @strUsername OR InstantASP_Users.Username = @strUsername) 
	END
END
RETURN
GO

SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_SelectWrappers
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_SelectWrappers 
AS

SET NOCOUNT ON 

/* Selects all available wrappers from the InstantASP_Wrappers table */

SELECT * FROM InstantASP_Wrappers (nolock) ORDER BY InstantASP_Wrappers.WrapperID DESC
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_ShrinkDatabase
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO


CREATE PROCEDURE iasp_sp_ShrinkDatabase
@strDatabaseName nvarchar(75),
@intPercentage int

/* Attempts to shrink the database */
AS DBCC SHRINKDATABASE (@strDatabaseName, @intPercentage)
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateAdminNotes
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_UpdateAdminNotes (
@intUserID int,
@strMessage ntext,
@intIdentity int output
)
AS

SET NOCOUNT ON 

/* Updates an admins notes within the InstantASP_Administrators table */

-- if notes do not already exists insert a new row
IF NOT EXISTS(SELECT AdministratorID FROM InstantASP_Administrators (nolock) WHERE InstantASP_Administrators.UserID = @intUserID)
BEGIN

	-- insert administrator with notes
	INSERT INTO InstantASP_Administrators (UserID, Notes) VALUES (@intUserID, @strMessage)

	-- return administrator record identity
	SET @intIdentity = (SELECT AdministratorID FROM InstantASP_Administrators WHERE InstantASP_Administrators.UserID = @intUserID)

END
ELSE -- we already have a record so simply update
BEGIN

	-- update existing record
	UPDATE InstantASP_Administrators SET InstantASP_Administrators.Notes = @strMessage
	WHERE (InstantASP_Administrators.UserID = @intUserID)

	-- return administrator record identity
	SET @intIdentity = (SELECT InstantASP_Administrators.AdministratorID FROM InstantASP_Administrators WHERE InstantASP_Administrators.UserID = @intUserID)

END
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateEmailAttempts
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_UpdateEmailAttempts (
@strIdentities nvarchar(2000)
) AS
SET NOCOUNT ON 
/* Accepts a comma delimited list of email identities and loops through this list incrementing retry count for failed emails */
DECLARE @Separator varchar(1)
DECLARE @Value varchar(255)
DECLARE @SepPosition int
DECLARE @LoopCount int
SET  @Separator = ","
SET @LoopCount = 0
SET @strIdentities = @strIdentities + @Separator
WHILE patindex('%' + @Separator + '%' , @strIdentities) <> 0 -- loop until we find no more separators
	BEGIN
	
		SET @SepPosition =  patindex('%' + @Separator + '%' , @strIdentities)  -- patindex matches the a pattern against a string
		SET @Value = left(@strIdentities, @SepPosition - 1) -- array value holds each single keyword
		
		/* Update email attempts */
		 UPDATE InstantASP_Emails SET InstantASP_Emails.EmailAttempts = InstantASP_Emails.EmailAttempts + 1 WHERE InstantASP_Emails.EmailID = @Value  
		
		SET @LoopCount = + 1 -- increment loop count		
		SET @strIdentities = stuff(@strIdentities, 1, @SepPosition, '') -- This replaces what we just processed with and empty string, pevents infinate loops
	END
GO

SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO
----------------------------------------------------
-- iasp_sp_UpdateLastLoginDate
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_UpdateLastLoginDate (
@intUserID int
)
AS

SET NOCOUNT ON 

UPDATE InstantASP_Users SET LastLoginDate = GetDate(), TotalVisits = TotalVisits + 1 WHERE UserID = @intUserID
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateLastLoginDateByEmailAddress
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_UpdateLastLoginDateByEmailAddress (
@strEmailAddress nvarchar(255)
)
AS

SET NOCOUNT ON 

UPDATE InstantASP_Users SET LastLoginDate = GetDate(), TotalVisits = TotalVisits + 1 WHERE EmailAddress = @strEmailAddress
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateRegistrationDetails
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO


CREATE PROCEDURE iasp_sp_UpdateRegistrationDetails (
@intUserID int,
@strUsername nvarchar(255),
@strEmailAddress nvarchar(255),
@strPassword nvarchar(255),
@intPrimaryRoleID int,
@strConfirmationCode nvarchar(25),
@intReturnCode int output
)
AS

SET NOCOUNT ON 

/* Updates a users primary login details */

-- Check to see if username already exists for any other user

IF EXISTS(SELECT UserID FROM InstantASP_Users WHERE Username = @strUsername AND UserID <> @intUserID)
BEGIN
	SET @intReturnCode = - 1
	RETURN
END 

-- Check to see if email address already exists for any other user

IF EXISTS(SELECT UserID FROM InstantASP_Users WHERE EmailAddress = @strEmailAddress AND UserID <> @intUserID)
BEGIN
	SET @intReturnCode = - 2
	RETURN
END 

-- update primary details

UPDATE InstantASP_Users SET Username = @strUsername, EmailAddress = @strEmailAddress, 
[Password] = @strPassword, PrimaryRoleID = @intPrimaryRoleID WHERE InstantASP_Users.UserID = @intUserID

-- if we have a new confirmation code update the ConfirmationCode column

IF (@strConfirmationCode != '')
BEGIN
	UPDATE InstantASP_Users SET  ConfirmationCode = @strConfirmationCode WHERE InstantASP_Users.UserID = @intUserID 
END

-- set returnCode ouput parameter to user identity

SET @intReturnCode = @intUserID
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateScheduledTaskGuid
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO



CREATE PROCEDURE iasp_sp_UpdateScheduledTaskGuid (
@intTaskID int,
@strTaskGuid nvarchar(75)
)
AS

SET NOCOUNT ON 

/* Updates a tasks GUID identifier used whilsts the task is running to ensure it's from the task manager */

-- update settings
UPDATE InstantASP_ScheduledTasks SET 
TaskGuid = @strTaskGuid
WHERE InstantASP_ScheduledTasks.TaskID = @intTaskID
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateScheduledTaskRunInformation
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_UpdateScheduledTaskRunInformation (
@intTaskID int,
@strTaskLastRunDate nvarchar(100),
@strTaskNextRunDate nvarchar(100)
)
AS

SET NOCOUNT ON 

/* Updates a tasks next run date within the InstantASP_ScheduledTasks table */

-- update settings
UPDATE InstantASP_ScheduledTasks SET 
TaskLastRunDate = @strTaskLastRunDate,
TaskNextRunDate = @strTaskNextRunDate
WHERE InstantASP_ScheduledTasks.TaskID = @intTaskID
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateSettings
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO


CREATE PROCEDURE iasp_sp_UpdateSettings (
@strCulture nvarchar(10),
@strCookiePrefix nvarchar(25),
@bitSMTPEmailEnabled bit,
@strSMTPServer nvarchar(255),
@strSMTPUsername nvarchar(255),
@strSMTPPassword nvarchar(255), 
@intSMTPPort int,
@strRssEncoding nvarchar(25),
@intRssCacheMinutes int,
@bitCompressOutput bit,
@bitRemoveSpanTags bit,
@dblTimeZoneOffset float,
@strDateTimeFormat nvarchar(1),
@intSMTPEmailType tinyint,
@intAccountActivationType tinyint,
@intDefaultUserRoleID int,
@intDefaultAwaitingApprovalRoleID int,
@strAnonymousUsername nvarchar(255),
@strAnonymousEmailAddress nvarchar(255),
@bitRequireConfirmationOnEmailUpdate bit,
@intFirstDayOfWeek tinyint,
@intLastLoginRefreshInterval smallint,
@bitHTTPHeadersNoCache bit,
@intLoginUsing tinyint,
@intEmailsPerCycle tinyint,
@intSearchType tinyint,
@intEmailAttempts tinyint,
@bitObserveDaylightSavingTime bit
)
AS
SET NOCOUNT ON 
/* Updates the global application settings within the InstantASP_Settings table */
-- update settings
UPDATE InstantASP_Settings SET 
Culture = @strCulture, 
CookiePrefix = @strCookiePrefix,
SMTPEmailEnabled = @bitSMTPEmailEnabled,
SMTPServer = @strSMTPServer,
SMTPUsername = @strSMTPUsername,
SMTPPassword = @strSMTPPassword,
SMTPPort = @intSMTPPort,
RssEncoding = @strRssEncoding,
RssCacheMinutes = @intRssCacheMinutes,
CompressOutput = @bitCompressOutput,
RemoveSpanTags = @bitRemoveSpanTags,
TimeZoneOffset = @dblTimeZoneOffset,
DateTimeFormat = @strDateTimeFormat,
SMTPEmailType = @intSMTPEmailType,
AccountActivationType = @intAccountActivationType,
DefaultUserRoleID = @intDefaultUserRoleID,
DefaultAwaitingApprovalRoleID = @intDefaultAwaitingApprovalRoleID,
AnonymousUsername = @strAnonymousUsername,
AnonymousEmailAddress = @strAnonymousEmailAddress,
RequireConfirmationOnEmailUpdate = @bitRequireConfirmationOnEmailUpdate,
FirstDayOfWeek = @intFirstDayOfWeek,
LastLoginRefreshInterval = @intLastLoginRefreshInterval,
HTTPHeadersNoCache = @bitHTTPHeadersNoCache,
LoginUsing = @intLoginUsing,
EmailsPerCycle = @intEmailsPerCycle,
SearchType = @intSearchType,
EmailAttempts = @intEmailAttempts,
ObserveDaylightSavingTime = @bitObserveDaylightSavingTime
GO


SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateUsersCulture
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_UpdateUsersCulture (
@intUserID int,
@strCulture nvarchar(20)
)
AS

SET NOCOUNT ON 

/* Updates a users culture setting within the InstantASP_Users table */

-- update settings
UPDATE InstantASP_Users SET 
Culture = @strCulture WHERE InstantASP_Users.UserID = @intUserID
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateUsersEmailAddress
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_UpdateUsersEmailAddress (
@intUserID int,
@strEmailAddress nvarchar(255),
@intReturnCode int output
)
AS

SET NOCOUNT ON 

/* Updates a users primary email address within the InstantASP_Users table */

-- check to see if email address already exists for any other user

IF EXISTS(SELECT InstantASP_Users.UserID FROM InstantASP_Users (nolock) WHERE (InstantASP_Users.EmailAddress = @strEmailAddress AND InstantASP_Users.UserID <> @intUserID))
BEGIN
	SET @intReturnCode = - 1
	RETURN
END 

-- update email address
UPDATE InstantASP_Users SET InstantASP_Users.EmailAddress = @strEmailAddress WHERE (InstantASP_Users.UserID = @intUserID)

-- set returnCode ouput parameter to user identity
SET @intReturnCode = @intUserID
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateUserSettings
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER ON
GO

GO

CREATE PROCEDURE iasp_sp_UpdateUserSettings (
@intUserID int,
@strCulture nvarchar(20),
@dblTimeZoneOffset float,
@bitObserveDaylightSavingTime bit,
@strDateTimeFormat nvarchar(1),
@intFirstDayOfWeek tinyint,
@intIdentity int output
)
AS
SET NOCOUNT ON 
/* Updates a users settings within the InstantASP_Users table */
-- update settings
UPDATE InstantASP_Users SET 
Culture = @strCulture, 
TimeZoneOffset = @dblTimeZoneOffset, 
ObserveDaylightSavingTime = @bitObserveDaylightSavingTime,
DateTimeFormat = @strDateTimeFormat,
FirstDayOfWeek = @intFirstDayOfWeek
WHERE InstantASP_Users.UserID = @intUserID
-- returns users identity, used to ensure transaction was successfull 
SET @intIdentity = (SELECT UserID FROM InstantASP_Users (nolock) WHERE InstantASP_Users.UserID = @intUserID)
GO

SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateUsersPassword
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_UpdateUsersPassword (
@intUserID int,
@strPassword nvarchar(255)
)
AS

SET NOCOUNT ON 

/* Updates a users password within the InstantASP_Users table */

-- update password
UPDATE InstantASP_Users SET InstantASP_Users.[Password] = @strPassword WHERE (InstantASP_Users.UserID = @intUserID)
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateUsersPrimaryRole
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_UpdateUsersPrimaryRole (
@intUserID int,
@intRoleID smallint,
@intReturnCode int output
)
AS

SET NOCOUNT ON 

/* Updates a users primary role within the InstantASP_Users table, also checks to ensure primary role relationship exists within InstantASP_UsersRoles */

-- update primary role
UPDATE InstantASP_Users SET InstantASP_Users.PrimaryRoleID = @intRoleID WHERE (InstantASP_Users.UserID = @intUserID)

-- deteremine if relationship exists
IF NOT EXISTS(SELECT InstantASP_UsersRoles.UserRoleID FROM InstantASP_UsersRoles (nolock) WHERE (InstantASP_UsersRoles.UserID = @intUserID AND InstantASP_UsersRoles.RoleID = @intRoleID))
BEGIN
	-- we did not find a relationship for the primary role so add one
	INSERT INTO InstantASP_UsersRoles (UserID, RoleID) VALUES (@intUserID, @intRoleID)
END

-- set returnCode ouput parameter to user identity
IF (@@ERROR <> 0)
BEGIN
	SET @intReturnCode = -1
END
ELSE
BEGIN
	SET @intReturnCode = @intUserID
END
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

----------------------------------------------------
-- iasp_sp_UpdateUsersUsername
----------------------------------------------------
GO
SET QUOTED_IDENTIFIER OFF
GO

GO


CREATE PROCEDURE iasp_sp_UpdateUsersUsername (
@intUserID int,
@strUsername nvarchar(255),
@intReturnCode int output
)
AS

SET NOCOUNT ON 

/* Updates a users username within the InstantASP_Users table */

-- check to see if username already exists for any other user

IF EXISTS(SELECT InstantASP_Users.UserID FROM InstantASP_Users (nolock) WHERE (Username = @strUsername AND InstantASP_Users.UserID <> @intUserID))
BEGIN
	SET @intReturnCode = - 1
	RETURN
END 

-- update username
UPDATE InstantASP_Users SET InstantASP_Users.Username = @strUsername WHERE (InstantASP_Users.UserID = @intUserID)

-- set returnCode ouput parameter to user identity
SET @intReturnCode = @intUserID
GO
SET ANSI_NULLS OFF
GO
SET QUOTED_IDENTIFIER OFF
GO

/******************************************************
  Stored procedures  End
******************************************************/
GO
/******************************************************
  Indexes  Begin
******************************************************/

GO
CREATE NONCLUSTERED INDEX IX_InstantASP_Users_PrimaryRoleID ON InstantASP_Users ( PrimaryRoleID ) 
WITH FILLFACTOR = 20

GO
CREATE NONCLUSTERED INDEX IX_InstantASP_Administrators_UserID ON InstantASP_Administrators ( UserID ) 

GO
CREATE NONCLUSTERED INDEX IX_InstantASP_Sessions_SessionIdentity ON InstantASP_Sessions ( SessionIdentity ) 
WITH FILLFACTOR = 10

GO
CREATE NONCLUSTERED INDEX IX_InstantASP_Users_EmailAddress ON InstantASP_Users ( EmailAddress ) 
WITH FILLFACTOR = 20

GO

/******************************************************
  Indexes  End
******************************************************/
GO


/******************************************************
  Insert data   Begin
******************************************************/

-----------------------------------------------------------
--Insert data into InstantASP_IgnoredSearchTerms
-----------------------------------------------------------
ALTER TABLE InstantASP_IgnoredSearchTerms NOCHECK CONSTRAINT ALL
GO

IF (IDENT_SEED('InstantASP_IgnoredSearchTerms') IS NOT NULL )	SET IDENTITY_INSERT InstantASP_IgnoredSearchTerms ON
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('2','\bof\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('3','\bthe\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('4','\bthis\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('5','\bthat\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('6','\ban\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('7','\ba\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('8','\bon\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('9','\bhave\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('10','\bfrom\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('11','\bjust\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('12','\bour\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('13','\bhas\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('14','\bhave\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('15','\bvery\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('16','\bare\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('17','\bto\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('18','\bwill\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('19','\bthem\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('20','\bso\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('21','\bsome\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('22','\bwhere\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('23','\bsee\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('24','\bwe\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('25','\buse\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('26','\bor\b')
INSERT INTO InstantASP_IgnoredSearchTerms (IgnoredSearchTermID,SearchText) VALUES('29','\band\b')
IF (IDENT_SEED('InstantASP_IgnoredSearchTerms') IS NOT NULL )	SET IDENTITY_INSERT InstantASP_IgnoredSearchTerms OFF

GO
ALTER TABLE InstantASP_IgnoredSearchTerms CHECK CONSTRAINT ALL
GO


-----------------------------------------------------------
--Insert data into InstantASP_Roles
-----------------------------------------------------------
ALTER TABLE InstantASP_Roles NOCHECK CONSTRAINT ALL
GO

IF (IDENT_SEED('InstantASP_Roles') IS NOT NULL )	SET IDENTITY_INSERT InstantASP_Roles ON
INSERT INTO InstantASP_Roles (RoleID,RoleName,RoleVisible,OpenMarkup,CloseMarkup,AnonymousRole,ActivationRole,MemberRole,ModeratorRole,AdministratorRole,BannedRole,RequiredRole) VALUES('1','Anonymous / Guest',1,'','',1,0,0,0,0,0,1)
INSERT INTO InstantASP_Roles (RoleID,RoleName,RoleVisible,OpenMarkup,CloseMarkup,AnonymousRole,ActivationRole,MemberRole,ModeratorRole,AdministratorRole,BannedRole,RequiredRole) VALUES('2','Forum Members',1,'','',0,0,1,0,0,0,1)
INSERT INTO InstantASP_Roles (RoleID,RoleName,RoleVisible,OpenMarkup,CloseMarkup,AnonymousRole,ActivationRole,MemberRole,ModeratorRole,AdministratorRole,BannedRole,RequiredRole) VALUES('3','Awaiting Activation',1,'<font color=''#A2A2A2''>','</font>',0,1,0,0,0,0,1)
INSERT INTO InstantASP_Roles (RoleID,RoleName,RoleVisible,OpenMarkup,CloseMarkup,AnonymousRole,ActivationRole,MemberRole,ModeratorRole,AdministratorRole,BannedRole,RequiredRole) VALUES('4','Moderators',1,'<b><i>','</i></b>',0,0,0,1,0,0,1)
INSERT INTO InstantASP_Roles (RoleID,RoleName,RoleVisible,OpenMarkup,CloseMarkup,AnonymousRole,ActivationRole,MemberRole,ModeratorRole,AdministratorRole,BannedRole,RequiredRole) VALUES('5','Administrators',1,'<b>','</b>',0,0,0,0,1,0,1)
INSERT INTO InstantASP_Roles (RoleID,RoleName,RoleVisible,OpenMarkup,CloseMarkup,AnonymousRole,ActivationRole,MemberRole,ModeratorRole,AdministratorRole,BannedRole,RequiredRole) VALUES('6','Banned Members',0,'<strike>','</strike>',0,0,0,0,0,1,1)
IF (IDENT_SEED('InstantASP_Roles') IS NOT NULL )	SET IDENTITY_INSERT InstantASP_Roles OFF

GO
ALTER TABLE InstantASP_Roles CHECK CONSTRAINT ALL
GO


-----------------------------------------------------------
--Insert data into InstantASP_Settings
-----------------------------------------------------------
ALTER TABLE InstantASP_Settings NOCHECK CONSTRAINT ALL
GO

IF (IDENT_SEED('InstantASP_Settings') IS NOT NULL )	SET IDENTITY_INSERT InstantASP_Settings ON
INSERT INTO InstantASP_Settings (SettingID,Culture,CookiePrefix,SMTPEmailEnabled,SMTPServer,SMTPUsername,SMTPPassword,SMTPPort,RssEncoding,RssCacheMinutes,CompressOutput,RemoveSpanTags,TimeZoneOffset,DateTimeFormat,SMTPEmailType,AuthenticationType,AccountActivationType,DefaultUserRoleID,DefaultAwaitingApprovalRoleID,AnonymousUsername,AnonymousEmailAddress,RequireConfirmationOnEmailUpdate,FirstDayOfWeek,LastLoginRefreshInterval,HttpHeadersNoCache,LoginUsing,EmailsPerCycle, SearchType) VALUES('1','en-us','IASP_',1,'localhost','','','25','UTF-8','20',0,0,-7,'G','1','1','1','2','3','Anonymous','you@yourdomain.com',0,'0','20',1,'1','50','1')
IF (IDENT_SEED('InstantASP_Settings') IS NOT NULL )	SET IDENTITY_INSERT InstantASP_Settings OFF

GO
ALTER TABLE InstantASP_Settings CHECK CONSTRAINT ALL
GO


-----------------------------------------------------------
--Insert data into InstantASP_Users
-----------------------------------------------------------
ALTER TABLE InstantASP_Users NOCHECK CONSTRAINT ALL
GO

IF (IDENT_SEED('InstantASP_Users') IS NOT NULL )	SET IDENTITY_INSERT InstantASP_Users ON
INSERT INTO InstantASP_Users (UserID,EmailAddress,Password,Username,PrimaryRoleID,Culture,IPAddress,TimeZoneOffset,DateTimeFormat,TotalVisits,CreatedDate,LastLoginDate,ConfirmationCode,FirstDayOfWeek) VALUES('1','admin@admin.com','bKj4eFBHbnA=','Admin','5','','127.0.0.1',-7,'G','0',GetDate(),GetDate(),'','0')
IF (IDENT_SEED('InstantASP_Users') IS NOT NULL )	SET IDENTITY_INSERT InstantASP_Users OFF

GO
ALTER TABLE InstantASP_Users CHECK CONSTRAINT ALL
GO


-----------------------------------------------------------
--Insert data into InstantASP_UsersRoles
-----------------------------------------------------------
ALTER TABLE InstantASP_UsersRoles NOCHECK CONSTRAINT ALL
GO

IF (IDENT_SEED('InstantASP_UsersRoles') IS NOT NULL )	SET IDENTITY_INSERT InstantASP_UsersRoles ON
INSERT INTO InstantASP_UsersRoles (UserRoleID,UserID,RoleID) VALUES('1','1','1')
INSERT INTO InstantASP_UsersRoles (UserRoleID,UserID,RoleID) VALUES('2','1','2')
INSERT INTO InstantASP_UsersRoles (UserRoleID,UserID,RoleID) VALUES('3','1','3')
INSERT INTO InstantASP_UsersRoles (UserRoleID,UserID,RoleID) VALUES('4','1','4')
INSERT INTO InstantASP_UsersRoles (UserRoleID,UserID,RoleID) VALUES('5','1','5')
INSERT INTO InstantASP_UsersRoles (UserRoleID,UserID,RoleID) VALUES('6','1','6')
IF (IDENT_SEED('InstantASP_UsersRoles') IS NOT NULL )	SET IDENTITY_INSERT InstantASP_UsersRoles OFF

GO
ALTER TABLE InstantASP_UsersRoles CHECK CONSTRAINT ALL
GO

/******************************************************
  Insert data  End
******************************************************/

