-- *****************************************************************************************
-- InstantKB.NET v1.2 to v1.3 upgrade script.
-- *****************************************************************************************
--
-- This script will convert an existing InstantKB v1.2 database to a InstantKB v1.3 database. 
-- To run this script please select your database within query analyzer and run this script.
-- The script will delete all your previous stored procs, add the new database tables and 
-- columns, and set-up new relationships with your existing tables. This will not delete 
-- any existing data.
--
-- IMPORTANT : We would strongly suggest you backup your current database before you run this 
-- script to easily revert should you experience any problems.
--
-- *****************************************************************************************

-------------------------------------------------------------------
-- Add New Tables
-------------------------------------------------------------------

CREATE TABLE [InstantKB_ArticlesRelArticles] (
	[RelatedArticleID] [int] IDENTITY (1, 1) NOT NULL ,
	[QuestionID] [int] NOT NULL ,
	[RelQuestionID] [int] NOT NULL 
) ON [PRIMARY]
GO

CREATE TABLE [InstantKB_ArticlesTreeNodes] (
	[NodeQuestionID] [int] IDENTITY (1, 1) NOT NULL ,
	[NodeID] [nvarchar] (75) NOT NULL ,
	[QuestionID] [int] NOT NULL 
) ON [PRIMARY]
GO

-------------------------------------------------------------------
-- Update Existing Tables 
-------------------------------------------------------------------

-- we have renamed InstantKB_AdminUsr to InstantKB_AdminUsers so create the new 
-- table and populate with content from InstantKB_AdminUsr then delete InstantKB_AdminUsr.

CREATE TABLE [InstantKB_AdminUsers] (
	[UserID] [int] IDENTITY (1, 1) NOT NULL ,
	[FullName] [nvarchar] (150) NOT NULL ,
	[Username] [nvarchar] (150) NOT NULL ,
	[Password] [nvarchar] (150) NOT NULL ,
	[EmailAddress] [nvarchar] (150) NULL ,
	[DateCreated] [smalldatetime] NULL ,
	[LastLogin] [smalldatetime] NULL 
) ON [PRIMARY]
GO

DECLARE @FullName [nvarchar] (150)  
DECLARE @Username [nvarchar] (150) 
DECLARE @Password [nvarchar] (150) 
DECLARE @EmailAddress [nvarchar] (150) 
DECLARE @DateCreated [smalldatetime]  
DECLARE @LastLogin [smalldatetime]  

DECLARE MSGCURSOR CURSOR FOR
SELECT FullName, Username, KBPassword, EmailAddress, DateCreated, LastLogin FROM InstantKB_AdminUsr

OPEN MSGCURSOR

FETCH NEXT FROM MSGCURSOR
INTO @FullName, @Username, @Password, @EmailAddress, @DateCreated, @LastLogin

WHILE @@FETCH_STATUS = 0
BEGIN

	INSERT INTO InstantKB_AdminUsers (FullName, Username, [Password], EmailAddress, DateCreated, LastLogin)
	VALUES (@FullName, @Username, @Password, @EmailAddress, @DateCreated, @LastLogin)

	FETCH NEXT FROM MSGCURSOR
	INTO @FullName, @Username, @Password, @EmailAddress, @DateCreated, @LastLogin

END

CLOSE MSGCURSOR
DEALLOCATE MSGCURSOR

DROP TABLE InstantKB_AdminUsr -- delete old admin table now we have copied data into new table

------------------------

-- loop through existing knowledge base articles and split each comma delimited list of 
-- related article identities into the new InstantKB_ArticlesRelArticles table

DECLARE @QuestionID1 [int]
DECLARE @RelatedArticles [nvarchar] (150) 

DECLARE @Delimiter1 char(1)
DECLARE @Separator_Position1 int 
DECLARE @Array_Value1 varchar(100) 
SET @Delimiter1 = ','

DECLARE MSGCURSOR1 CURSOR FOR
SELECT QuestionID, RelatedArticles FROM InstantKB_Articles

OPEN MSGCURSOR1

FETCH NEXT FROM MSGCURSOR1
INTO @QuestionID1, @RelatedArticles

WHILE @@FETCH_STATUS = 0
BEGIN
	
	IF (@RelatedArticles IS NOT NULL)
	BEGIN

		SET @RelatedArticles = @RelatedArticles + @Delimiter1
		WHILE patindex('%' + @Delimiter1  + '%' , @RelatedArticles) <> 0 
		BEGIN
		
			SET @Separator_Position1 =  patindex('%' +  @Delimiter1 + '%' , @RelatedArticles) 
			SET @Array_Value1 = left(@RelatedArticles, @Separator_Position1 - 1) 
		
			INSERT INTO InstantKB_ArticlesRelArticles (QuestionID, RelQuestionID) 
			VALUES (@QuestionID1, @Array_Value1)
						
			SET @RelatedArticles = stuff(@RelatedArticles, 1, @Separator_Position1, '')
			
		END

	END
	
	FETCH NEXT FROM MSGCURSOR1
	INTO @QuestionID1, @RelatedArticles

END

CLOSE MSGCURSOR1
DEALLOCATE MSGCURSOR1

-- delete related articles column now we have copied data to new table
ALTER TABLE [InstantKB_Articles] DROP COLUMN RelatedArticles 
GO

------------------------

-- loop through existing knowledge base articles grab the node id for the article
-- then add this information to the new InstantKB_ArticlesTreeNodes

DECLARE @QuestionID [int]
DECLARE @NodeID [nvarchar] (150) 

DECLARE MSGCURSOR2 CURSOR FOR
SELECT NodeID, QuestionID FROM InstantKB_Articles

OPEN MSGCURSOR2

FETCH NEXT FROM MSGCURSOR2
INTO @NodeID, @QuestionID

WHILE @@FETCH_STATUS = 0
BEGIN
	
	IF (@NodeID IS NOT NULL)
	BEGIN
		INSERT INTO InstantKB_ArticlesTreeNodes (NodeID, QuestionID) VALUES (@NodeID, @QuestionID)
	END
	
	FETCH NEXT FROM MSGCURSOR2
	INTO @NodeID, @QuestionID

END

CLOSE MSGCURSOR2
DEALLOCATE MSGCURSOR2

------------------------

-- delete nodeid column now we have copied data to new table
ALTER TABLE [InstantKB_Articles] DROP COLUMN NodeID 
GO

-- add new columns to article table
ALTER TABLE [InstantKB_Articles] ADD SkillLevel tinyint
GO

-- add review date column
ALTER TABLE [InstantKB_Articles] ADD ReviewDate smalldatetime
GO

-- alter nodeid column within instantkb_tree to non nullable
ALTER TABLE [InstantKB_Tree] ALTER COLUMN [NodeID] [nvarchar] (75) NOT NULL 
GO

-- update any empty values for the ParentNodeID column
UPDATE InstantKB_Tree SET ParentNodeID = null where ParentNodeID = ''
GO

-------------------------------------------------------------------
-- Add Indexes and Primary Keys 
-------------------------------------------------------------------

ALTER TABLE [InstantKB_AdminUsers] WITH NOCHECK ADD 
	CONSTRAINT [PK_InstantKB_AdminUsers] PRIMARY KEY  CLUSTERED 
	(
		[UserID]
	) WITH  FILLFACTOR = 30  ON [PRIMARY] 
GO

ALTER TABLE [InstantKB_Articles] WITH NOCHECK ADD 
	CONSTRAINT [PK_InstantKB_Articles] PRIMARY KEY  CLUSTERED 
	(
		[QuestionID]
	) WITH  FILLFACTOR = 50  ON [PRIMARY] 
GO

ALTER TABLE [InstantKB_ArticlesRelArticles] WITH NOCHECK ADD 
	CONSTRAINT [PK_InstantKB_ArticlesRelArticles] PRIMARY KEY  CLUSTERED 
	(
		[RelatedArticleID]
	) WITH  FILLFACTOR = 30  ON [PRIMARY] 
GO

ALTER TABLE [InstantKB_ArticlesTreeNodes] WITH NOCHECK ADD 
	CONSTRAINT [PK_InstantKB_ArticlesTreeNodes] PRIMARY KEY  CLUSTERED 
	(
		[NodeQuestionID]
	) WITH  FILLFACTOR = 50  ON [PRIMARY] 
GO

ALTER TABLE [InstantKB_Attachments] WITH NOCHECK ADD 
	CONSTRAINT [PK_InstantKB_Attachments] PRIMARY KEY  CLUSTERED 
	(
		[AttachmentID]
	) WITH  FILLFACTOR = 30  ON [PRIMARY] 
GO

ALTER TABLE [InstantKB_Comments] WITH NOCHECK ADD 
	CONSTRAINT [PK_InstantKB_Comments] PRIMARY KEY  CLUSTERED 
	(
		[CommentID]
	) WITH  FILLFACTOR = 30  ON [PRIMARY] 
GO

ALTER TABLE [InstantKB_Ratings] WITH NOCHECK ADD 
	CONSTRAINT [PK_InstantKB_Ratings] PRIMARY KEY  CLUSTERED 
	(
		[RatingID]
	) WITH  FILLFACTOR = 30  ON [PRIMARY] 
GO

ALTER TABLE [InstantKB_Roles] WITH NOCHECK ADD 
	CONSTRAINT [PK_InstantKB_Roles] PRIMARY KEY  CLUSTERED 
	(
		[RoleID]
	) WITH  FILLFACTOR = 30  ON [PRIMARY] 
GO

ALTER TABLE [InstantKB_Tree] WITH NOCHECK ADD 
	CONSTRAINT [PK_InstantKB_Tree] PRIMARY KEY  CLUSTERED 
	(
		[NodeID]
	) WITH  FILLFACTOR = 50  ON [PRIMARY] 
GO

ALTER TABLE [InstantKB_TreeRoles] WITH NOCHECK ADD 
	CONSTRAINT [PK_InstantKB_TreeRoles] PRIMARY KEY  CLUSTERED 
	(
		[NodeRoleID]
	) WITH  FILLFACTOR = 50  ON [PRIMARY] 
GO

ALTER TABLE [InstantKB_UserRoles] WITH NOCHECK ADD 
	CONSTRAINT [PK_InstantKB_UserRoles] PRIMARY KEY  CLUSTERED 
	(
		[UserRoleID]
	) WITH  FILLFACTOR = 20  ON [PRIMARY] 
GO

ALTER TABLE [InstantKB_Users] WITH NOCHECK ADD 
	CONSTRAINT [PK_InstantKB_Users] PRIMARY KEY  CLUSTERED 
	(
		[UserID]
	) WITH  FILLFACTOR = 30  ON [PRIMARY] 
GO

-- set default values

ALTER TABLE [InstantKB_AdminUsers] WITH NOCHECK ADD 
	CONSTRAINT [DF_InstantKB_AdminUsers_DateCreated] DEFAULT (getdate()) FOR [DateCreated],
	CONSTRAINT [DF_InstantKB_AdminUsers_LastLogin] DEFAULT (getdate()) FOR [LastLogin]
GO

ALTER TABLE [InstantKB_Articles] WITH NOCHECK ADD 
	CONSTRAINT [DF_InstantKB_Articles_SuggestedFlag] DEFAULT (0) FOR [SuggestedFlag],
	CONSTRAINT [DF_InstantKB_Articles_DifficultyRating] DEFAULT (0) FOR [SkillLevel],
	CONSTRAINT [DF_InstantKB_Articles_NoOfViews] DEFAULT (0) FOR [NoOfViews],
	CONSTRAINT [DF_InstantKB_Articles_DateStamp] DEFAULT (getdate()) FOR [DateStamp],
	CONSTRAINT [DF_InstantKB_Articles_PublishToWeb] DEFAULT (0) FOR [PublishToWeb]
GO

ALTER TABLE [InstantKB_Attachments] WITH NOCHECK ADD 
	CONSTRAINT [DF_InstantKB_Attachments_DateStamp] DEFAULT (getdate()) FOR [DateStamp]
GO

ALTER TABLE [InstantKB_Ratings] WITH NOCHECK ADD 
	CONSTRAINT [DF_InstantKB_Ratings_Rating] DEFAULT (0) FOR [Rating]
GO

ALTER TABLE [InstantKB_Tree] WITH NOCHECK ADD 
	CONSTRAINT [DF_InstantKB_Tree_SortOrder] DEFAULT (0) FOR [SortOrder]
GO

-- create non-clustered indexes

 CREATE  INDEX [QuestionID] ON [InstantKB_ArticlesRelArticles]([QuestionID]) ON [PRIMARY]
GO

 CREATE  INDEX [RelQuestionID] ON [InstantKB_ArticlesRelArticles]([RelQuestionID]) ON [PRIMARY]
GO

 CREATE  INDEX [QuestionID] ON [InstantKB_ArticlesTreeNodes]([QuestionID]) ON [PRIMARY]
GO

 CREATE  INDEX [NodeID] ON [InstantKB_ArticlesTreeNodes]([NodeID]) ON [PRIMARY]
GO

 CREATE  INDEX [QuestionID] ON [InstantKB_Attachments]([QuestionID]) ON [PRIMARY]
GO

 CREATE  INDEX [QuestionID] ON [InstantKB_Ratings]([QuestionID]) ON [PRIMARY]
GO

 CREATE  INDEX [Rating] ON [InstantKB_Ratings]([Rating]) ON [PRIMARY]
GO

 CREATE  INDEX [NodeID] ON [InstantKB_TreeRoles]([NodeID]) ON [PRIMARY]
GO

 CREATE  INDEX [RoleID] ON [InstantKB_TreeRoles]([RoleID]) ON [PRIMARY]
GO

 CREATE  INDEX [UserID] ON [InstantKB_UserRoles]([UserID]) ON [PRIMARY]
GO

 CREATE  INDEX [RoleID] ON [InstantKB_UserRoles]([RoleID]) ON [PRIMARY]
GO

-------------------------------------------------------------------
-- Remove all the old v1.2 stored procedures:
-------------------------------------------------------------------

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AddComments]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AddComments]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AddRating]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AddRating]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminAddAdmin]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminAddAdmin]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminAddArticle]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminAddArticle]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminAddAttachment]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminAddAttachment]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminAddCategory]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminAddCategory]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminAddCategoryRoles]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminAddCategoryRoles]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminAddUser]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminAddUser]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminAddUserRole]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminAddUserRole]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminAddUserRoles]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminAddUserRoles]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminApproveComment]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminApproveComment]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminDeleteArticle]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminDeleteArticle]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminDeleteAttachment]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminDeleteAttachment]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminDeleteCategory]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminDeleteCategory]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminDeleteComment]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminDeleteComment]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminDeleteRole]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminDeleteRole]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminDeleteTreeRoles]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminDeleteTreeRoles]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminDeleteUser]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminDeleteUser]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminDeleteUserRoles]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminDeleteUserRoles]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminGetAdminDetails]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminGetAdminDetails]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminGetCategoryInformation]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminGetCategoryInformation]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminGetNodes]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminGetNodes]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminGetUserDetails]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminGetUserDetails]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminLogin]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminLogin]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminSearchArticles]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminSearchArticles]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminUpdateAdmin]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminUpdateAdmin]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminUpdateArticle]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminUpdateArticle]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminUpdateCategoryInformation]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminUpdateCategoryInformation]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminUpdateComment]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminUpdateComment]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminUpdateLastLogin]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminUpdateLastLogin]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminUpdateRole]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminUpdateRole]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_AdminUpdateUser]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_AdminUpdateUser]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_CheckUserRolesAgainstNodeID]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_CheckUserRolesAgainstNodeID]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetAdminLastLogin]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetAdminLastLogin]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetAllNodeIDs]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetAllNodeIDs]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetArticle]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetArticle]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetArticleCount]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetArticleCount]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetArticleRating]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetArticleRating]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetArticleTitle]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetArticleTitle]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetArticles]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetArticles]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetAttachments]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetAttachments]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetAuthorEmail]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetAuthorEmail]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetAuthorName]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetAuthorName]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetComments]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetComments]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetKBStatisics]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetKBStatisics]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetLatestArticles]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetLatestArticles]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetNavBarLinkInfo]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetNavBarLinkInfo]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetNodeIDForArticle]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetNodeIDForArticle]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetNodeRoleRelationships]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetNodeRoleRelationships]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetNodeTitle]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetNodeTitle]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetNodes]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetNodes]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetParentNodeIDs]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetParentNodeIDs]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetPopularArticles]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetPopularArticles]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetRoles]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetRoles]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetUserRoleRelationships]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetUserRoleRelationships]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetUserRoles]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetUserRoles]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetUserRolesForDDL]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetUserRolesForDDL]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_GetUsers]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_GetUsers]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_IncrementArticleViews]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_IncrementArticleViews]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_ObtainDatabaseInformation]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_ObtainDatabaseInformation]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_doSearch]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_doSearch]
GO

if exists (select * from dbo.sysobjects where id = object_id(N'[ikb_sp_hasChildren]') and OBJECTPROPERTY(id, N'IsProcedure') = 1)
drop procedure [ikb_sp_hasChildren]
GO

-------------------------------------------------------------------
-- Now add all the new v1.3 stored procedures
-------------------------------------------------------------------



CREATE Procedure ikb_sp_AddComments (
@comments nvarchar(1000),
@articleID int
) AS
INSERT INTO InstantKB_Comments (articleID, Comments, DateStamp, Approved) VALUES (@articleID, @comments, GetDate(), 0)
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AddRating (
@rating int,
@articleID int
) AS
INSERT INTO InstantKB_Ratings (rating, questionID) VALUES (@rating, @articleID)
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminAddAdmin(
@strFullName nvarchar(150),
@strEmailAddress nvarchar(150),
@strUserName nvarchar(150),
@strPassword nvarchar(150)
)  AS

IF (@strFullName = '')
BEGIN
	SET @strFullName = null
END

IF (@strEmailAddress = '')
BEGIN
	SET @strEmailAddress = null
END

IF (@strUserName = '')
BEGIN
	SET @strUserName = null
END

IF (@strPassword = '')
BEGIN
	SET @strPassword = null
END

INSERT INTO InstantKB_AdminUsers (FullName, EmailAddress, Username, [Password], DateCreated) VALUES (@strFullName, @strEmailAddress, @strUserName, @strPassword, GetDate())
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminAddArticle (
@strArticleQuestion nvarchar(500),
@strArticleKeywords nvarchar(500),
@strNodeIDs varchar(500),
@strSuggestedBy nvarchar(150),
@strSuggestedByEmail nvarchar(150),
@intSuggestedFlag bit,
@strWYSIWYG ntext,
@strRelatedLinks nvarchar(800),
@strArticleStatus nvarchar(1),
@intAdminID TinyInt,
@strArticleType nvarchar(100),
@intSkillLevel TinyInt,
@dtReviewDate varchar(100),
@intArticleID Int out
)  AS
If (@strArticleKeywords = "")
BEGIN
	SET @strArticleKeywords = null
END
If (@strArticleQuestion = "")
BEGIN
	SET @strArticleQuestion = null
END
If (@strSuggestedBy = "")
BEGIN
	SET @strSuggestedBy = null
END
If (@strSuggestedByEmail = "")
BEGIN
	SET @strSuggestedByEmail = null
END
If (@intSuggestedFlag = "")
BEGIN
	SET @intSuggestedFlag = 0
END
If (@strRelatedLinks = "")
BEGIN
	SET @strRelatedLinks = null
END
If (@strArticleStatus = "")
BEGIN
	SET @strArticleStatus = null
END
If (@intAdminID = "")
BEGIN
	SET @intAdminID = null
END
If (@strArticleType = "")
BEGIN
	SET @strArticleType = null
END
If (@dtReviewDate = "")
BEGIN
	SET @dtReviewDate = null
END
ELSE
BEGIN
	SET @dtReviewDate = CONVERT(varchar,@dtReviewDate,100)
END

INSERT INTO InstantKB_Articles 
(QuestionText, Keywords, AnswerText, SuggestedBy, SuggestedByEmail, SuggestedFlag, RelatedLinks, DateStamp, LastModified, NoOfViews, PublishToWeb, AdminID, ArticleType, SkillLevel, ReviewDate)
VALUES (@strArticleQuestion, @strArticleKeywords, @strWYSIWYG, @strSuggestedBy, @strSuggestedByEmail, @intSuggestedFlag, @strRelatedLinks, 
GetDate(), GetDate(), 1, @strArticleStatus, @intAdminID, @strArticleType, @intSkillLevel, @dtReviewDate);

SET @intArticleID = @@identity;

-- Parse the comma-separated values list of NodeIDs and insert a row in the InstantKB_ArticlesTreeNodes table for each of them
DECLARE @intPos int, @NodeID nvarchar(75);
SET @strNodeIDs = LTRIM(RTRIM(@strNodeIDs))+ ',';
SET @intPos = CHARINDEX(',', @strNodeIDs, 1);
IF REPLACE(@strNodeIDs, ',', '') <> ''
BEGIN
	WHILE @intPos > 0
	BEGIN
		SET @NodeID = LTRIM(RTRIM(LEFT(@strNodeIDs, @intPos - 1)))
		IF @NodeID <> ''
		BEGIN
			INSERT INTO InstantKB_ArticlesTreeNodes (QuestionID, NodeID) VALUES (@intArticleID, @NodeID);
		END
		-- Advance the position to the next NodeID
		SET @strNodeIDs = RIGHT(@strNodeIDs, LEN(@strNodeIDs) - @intPos)
		SET @intPos = CHARINDEX(',', @strNodeIDs, 1)
	END
END	

return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminAddAttachment (
@FileName nVarChar(150),
@articleID Int
)  AS
INSERT INTO InstantKB_Attachments (questionID, Document, DateStamp) VALUES (@articleID,@FileName, GetDate())
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminAddCategory(
@nodeID nvarchar(75),
@nodeName nvarchar(300),
@parentNodeID nvarchar(75),
@sortOrder int,
@customIcon nvarchar(75)
)  AS
If (@customIcon = "")
BEGIN
	SET @customIcon = null
END
If (@parentNodeID = "")
BEGIN
	SET @parentNodeID = null
END
INSERT INTO InstantKB_Tree (NodeID, NodeName, ParentNodeID, SortOrder, IconPath) VALUES (@nodeID,@nodeName, @parentNodeID, @sortOrder, @customIcon)
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminAddCategoryRoles (
@nodeID nvarchar(75),
@roleID int
)  AS
INSERT INTO InstantKB_TreeRoles (NodeID, RoleID) VALUES (@nodeID, @roleID)
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminAddUser (
@Username nvarchar(255),
@Password nvarchar(255),
@UserID int output
) AS
IF (@Username = "")
BEGIN 
	SET @Username = null
END
IF (@Password = "")
BEGIN 
	SET @Password = null
END
INSERT INTO InstantKB_Users (Username, [Password]) VALUES (@Username, @Password)
SET @UserID = (SELECT @@IDENTITY)
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminAddUserRole (
@RoleName nvarchar(255)
)  AS
INSERT INTO InstantKB_Roles (RoleDescription) VALUES (@RoleName)
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminAddUserRoles (
@UserID int,
@RoleID int
)  AS
INSERT INTO InstantKB_UserRoles (UserID, RoleID) VALUES (@UserID, @RoleID)
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminApproveComment (
@commentID Int
)  AS
UPDATE InstantKB_Comments SET Approved = 1 WHERE commentID = @commentID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE PROCEDURE ikb_sp_AdminClearArticleRelationships @ArticleID int AS
	DELETE FROM InstantKB_ArticlesRelArticles WHERE QuestionID = @ArticleID;
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminDeleteAdministrator (
@AdminID int
)  AS
DELETE FROM InstantKB_AdminUsers WHERE UserID = @AdminID
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminDeleteArticle (
@articleID Int
)  AS
DELETE FROM InstantKB_Articles WHERE QuestionID = @articleID;
DELETE FROM InstantKB_ArticlesRelArticles WHERE QuestionID = @articleID OR RelQuestionID = @articleID;
DELETE FROM InstantKB_Attachments WHERE QuestionID = @articleID;
DELETE FROM InstantKB_ArticlesTreeNodes WHERE QuestionID = @articleID;
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminDeleteAttachment (
@attachmentID int,
@filename nvarchar(150) output
) AS
    
SET @filename = (SELECT document FROM InstantKB_Attachments WHERE AttachmentID =  @attachmentID)
DELETE FROM InstantKB_Attachments WHERE AttachmentID = @attachmentID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminDeleteCategory (
@categoryID nvarchar(75)
)  AS

DELETE FROM InstantKB_Tree WHERE NodeID = @categoryID
--DELETE FROM InstantKB_TreeRoles WHERE NodeID = @categoryID

return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminDeleteComment (
@commentID Int
)  AS
DELETE FROM InstantKB_Comments WHERE commentID = @commentID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminDeleteRole (
@RoleID Int
)  AS
DELETE FROM InstantKB_Roles WHERE RoleID = @RoleID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminDeleteTreeRoles (
@NodeID nvarchar(75)
)  AS
DELETE FROM InstantKB_TreeRoles WHERE NodeID = @NodeID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminDeleteUser (
@UserID int
)  AS
DELETE FROM InstantKB_Users WHERE UserID = @UserID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminDeleteUserRoles (
@UserID int
)  AS
DELETE FROM InstantKB_UserRoles WHERE UserID = @UserID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminGetAdminDetails (
@AdminID int
) AS
    
SELECT FullName, EmailAddress, Username, [Password] FROM InstantKB_AdminUsers WHERE UserID = @AdminID
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminGetAdminUsers
AS
SELECT FullName, UserID FROM InstantKB_AdminUsers ORDER BY UserID ASC
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminGetCategoryInformation (
@nodeID nvarchar(75)
) AS
SELECT NodeName, ParentNodeID, SortOrder, IconPath FROM InstantKb_Tree WHERE NodeID = @nodeID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminGetNodes (
@ParentID nvarchar(75)
) AS

If (@ParentID = "")
BEGIN
	SET @ParentID = null
END

SELECT * FROM InstantKB_Tree WHERE ParentNodeID = @ParentID ORDER BY SortOrder DESC, NodeName ASC
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminGetUserDetails (
@UserID int
) AS
    
SELECT* FROM InstantKB_Users WHERE UserID = @UserID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminGetUsers
AS
SELECT * FROM InstantKB_Users ORDER BY UserName ASC
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminLogin (
@UserName nvarchar(255),
@PassWord nvarchar(255),
@UserID int output
) AS
    
IF NOT EXISTS(SELECT * FROM InstantKB_AdminUsers WHERE Username = @UserName AND [Password] = @PassWord)
	BEGIN
		SET @UserID = 0
	END 
	ELSE
	BEGIN 
		SET @UserID = (SELECT UserID FROM InstantKB_AdminUsers WHERE Username = @UserName AND [Password] = @PassWord)
	END
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminSearch (
@strQuery nvarchar(4000), -- keywords passed into the procedure
@strNodeID nvarchar(75), -- the node to search for within the knowledge base
@strUsing nvarchar(50), -- type of search to perform - options are ALL, ANY and EXact
@strArticleStatus nvarchar(50), -- field to search options are full title and article id
@strMaximumAge datetime, -- daterange to search
@strReviewDate varchar(100)
) AS
DECLARE @SQL varchar(4000)
DECLARE @separator_position int -- This is used to locate each separator character
DECLARE @array_value varchar(4000) -- this holds each array value as it is returned
DECLARE @LoopCount int -- used to count loops when ANY type search is selected
DECLARE @separator varchar(1) -- separator used to split query at spaces when searching for any word
DECLARE @userRoleDelimiter varchar(1)  -- separator used to split user-role ids
DECLARE @LOperator varchar(5)
SET @separator = " " 
SET @LoopCount = 0 
If (@strReviewDate = "")
BEGIN
	SET @strReviewDate = null
END
ELSE
BEGIN
	SET @strReviewDate = CONVERT(varchar,@strReviewDate,100)
END
SELECT @SQL = 'SELECT DISTINCT InstantKB_Articles.QuestionID, InstantKB_Articles.QuestionText, InstantKB_Articles.Keywords, 
InstantKB_Articles.ArticleType, InstantKB_Articles.LastModified, InstantKB_Articles.DateStamp, InstantKB_Articles.PublishToWeb, 
InstantKB_Articles.SuggestedFlag FROM InstantKB_Articles INNER JOIN InstantKB_ArticlesTreeNodes ON InstantKB_Articles.QuestionID = InstantKB_ArticlesTreeNodes.QuestionID '

IF (@strUsing = "ex") -- all keywords search
BEGIN
	
SELECT @SQL = @SQL + ' WHERE (InstantKB_Articles.QuestionID LIKE ''%' + @strQuery  + '%'' OR InstantKB_Articles.QuestionText LIKE ''%' + @strQuery  + '%'' or 
InstantKB_Articles.AnswerText LIKE  ''%' + @strQuery + '%'' or InstantKB_Articles.Keywords LIKE  ''%' + @strQuery + '%'' or InstantKB_Articles.ArticleType LIKE  ''%' + @strQuery + '%'')' 
		
END
IF ((@strUsing = "any" or @strUsing = "all")) -- any word within search string - split words
BEGIN
SELECT @SQL = @SQL + ' WHERE ('
	SET @strQuery = @strQuery + @separator
		
	-- Loop through the string searching for separtor characters
	WHILE patindex('%' + @separator + '%' , @strQuery) <> 0 -- loop until we find no more separators
	BEGIN
		If (@strUsing = "any") -- if its not the first loop append OR to SQL statement
		BEGIN
			SET @LOperator = " OR "
		END
		ELSE BEGIN
			SET @LOperator = " AND "
		END
		SET @separator_position =  patindex('%' + @separator + '%' , @strQuery)  -- patindex matches the a pattern against a string
		SET @array_value = left(@strQuery, @separator_position - 1) -- array value holds each single keyword
	
		If (@LoopCount <> 0) -- if its not the first loop append OR to SQL statement
		BEGIN
			SELECT @SQL = @SQL + @LOperator
		END
		-- @array_value holds the value of this element of the array
		
		SELECT @SQL = @SQL + '(InstantKB_Articles.QuestionID LIKE  ''%' + @array_value + '%'' OR
		InstantKB_Articles.QuestionText LIKE ''%' + @array_value + '%'' OR InstantKB_Articles.AnswerText LIKE ''%' + @array_value + '%'' OR 
		InstantKB_Articles.ArticleType LIKE  ''%' + @array_value + '%'' OR InstantKB_Articles.Keywords LIKE  ''%' + @array_value + '%'')' 
			
		SET @LoopCount = + 1 -- increment loop count		
		SET @strQuery = stuff(@strQuery, 1, @separator_position, '') -- This replaces what we just processed with and empty string, pevents infinate loops
	END
END
	SELECT @SQL = @SQL + ') '

IF Len(@strNodeID) > 0
BEGIN
	SELECT @SQL = @SQL + 'AND (InstantKB_ArticlesTreeNodes.NodeID=''' + @strNodeID + ''') '
END

SELECT @SQL = @SQL + 'AND ('

IF (@strArticleStatus = 1)
BEGIN
	SELECT @SQL = @SQL + 'InstantKB_Articles.PublishToWeb = 1 AND'
END
ELSE IF (@strArticleStatus = 2)
BEGIN
	SELECT @SQL = @SQL + 'InstantKB_Articles.PublishToWeb = 0 AND'
END

IF (@strReviewDate <> null)
BEGIN
SELECT @SQL = @SQL + ' (InstantKB_Articles.ReviewDate >= ''' + CAST(GetDate() AS nVarChar(255)) + ''' AND InstantKB_Articles.ReviewDate <= ''' + @strReviewDate + ''') AND'
END

SELECT @SQL = @SQL + ' InstantKB_Articles.DateStamp >= ''' + 
cast(@strMaximumAge as varchar) + ''') AND (InstantKB_Articles.SuggestedFlag = 0) ORDER BY InstantKB_Articles.LastModified DESC, InstantKB_Articles.DateStamp DESC' 

Exec (@SQL)
return
GO




SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO

CREATE PROCEDURE ikb_sp_AdminSetArticleRelationship (
@ArticleID int, 
@RelArticleID int
) AS
	INSERT INTO InstantKB_ArticlesRelArticles (QuestionID, RelQuestionID)
	VALUES (@ArticleID, @RelArticleID)
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminUpdateAdmin (
@strFullName nvarchar(150),
@strEmailAddress nvarchar(150),
@strUserName nvarchar(150),
@strPassword nvarchar(150),
@intAdminID int
)  AS

IF (@strFullName = '')
BEGIN
	SET @strFullName = null
END

IF (@strEmailAddress = '')
BEGIN
	SET @strEmailAddress = null
END

IF (@strUserName = '')
BEGIN
	SET @strUserName = null
END

IF (@strPassword = '')
BEGIN
	SET @strPassword = null
END

UPDATE InstantKB_AdminUsers
SET FullName = @strFullName,
EmailAddress = @strEmailAddress,
Username = @strUserName,
[Password] = @strPassword
WHERE UserID = @intAdminID
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminUpdateArticle (
@strArticleQuestion nvarchar(500),
@strArticleKeywords nvarchar(500),
@strNodeIDs varchar(500),
@strSuggestedBy nvarchar(150),
@strSuggestedByEmail nvarchar(150),
@strWYSIWYG ntext,
@strRelatedLinks nvarchar(800),
@strArticleStatus nvarchar(1),
@strPrevSuggestionBy nvarchar(10),
@strArticleType nvarchar(100),
@intAdminID TinyInt,
@intQuestionID Int,
@intSkillLevel TinyInt,
@dtReviewDate VarChar(100)
)  AS
DECLARE @suggestedFlag int
If (@strArticleQuestion = "")
BEGIN
	SET @strArticleQuestion = null
END
If (@strArticleKeywords = "")
BEGIN
	SET @strArticleKeywords = null
END
If (@strNodeIDs = "")
BEGIN
	SET @strNodeIDs = null
END
If (@strSuggestedBy = "")
BEGIN
	SET @strSuggestedBy = null
END
If (@strSuggestedByEmail = "")
BEGIN
	SET @strSuggestedByEmail = null
END
If (@strRelatedLinks = "")
BEGIN
	SET @strRelatedLinks = null
END
If (@strArticleStatus = "")
BEGIN
	SET @strArticleStatus = null
END
If (@strArticleType = "")
BEGIN
	SET @strArticleType = null
END
If (@intAdminID = "")
BEGIN
	SET @intAdminID = null
END
IF (@strArticleStatus = "0" AND @strPrevSuggestionBy = "True") -- if article is not published within kb keep suggestedFlag as true
BEGIN
	SET @SuggestedFlag = 1
END
	ELSE -- only set suggested to false when a question has been approved
BEGIN
	SET @SuggestedFlag = 0
END
If (@dtReviewDate = "")
BEGIN
	SET @dtReviewDate = null
END
ELSE
BEGIN
	SET @dtReviewDate = CONVERT(varchar,@dtReviewDate,100)
END

UPDATE InstantKB_Articles SET QuestionText = @strArticleQuestion,
Keywords = @strArticleKeywords,
AnswerText = @strWYSIWYG,
SuggestedBy = @strSuggestedBy,
SuggestedByEmail = @strSuggestedByEmail,
SuggestedFlag = @suggestedFlag,
RelatedLinks = @strRelatedLinks,
LastModified = GetDate(),
PublishToWeb = @strArticleStatus,
AdminID = @intAdminID,
ArticleType = @strArticleType,
SkillLevel = @intSkillLevel,
ReviewDate = @dtReviewDate
WHERE QuestionID = @intQuestionID;
DELETE FROM InstantKB_ArticlesTreeNodes
WHERE QuestionID = @intQuestionID;

-- Parse the comma-separated values list of NodeIDs and insert a row in the InstantKB_ArticlesTreeNodes table for each of them

DECLARE @intPos int, @NodeID nvarchar(75);
SET @strNodeIDs = LTRIM(RTRIM(@strNodeIDs))+ ',';
SET @intPos = CHARINDEX(',', @strNodeIDs, 1);
IF REPLACE(@strNodeIDs, ',', '') <> ''
BEGIN
	WHILE @intPos > 0
	BEGIN
		SET @NodeID = LTRIM(RTRIM(LEFT(@strNodeIDs, @intPos - 1)))
		IF @NodeID <> ''
		BEGIN
			INSERT INTO InstantKB_ArticlesTreeNodes (QuestionID, NodeID) VALUES (@intQuestionID, @NodeID);
		END
		-- Advance the position to the next NodeID
		SET @strNodeIDs = RIGHT(@strNodeIDs, LEN(@strNodeIDs) - @intPos)
		SET @intPos = CHARINDEX(',', @strNodeIDs, 1)
	END
END	

return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminUpdateCategoryInformation (
@NodeID varchar(75),
@NodeName varchar(300),
@ParentNodeID varchar(75),
@SortOrder int,
@IconPath varchar(75)
)  AS
DECLARE @suggestedFlag int
If (@NodeName = "")
BEGIN
	SET @NodeName = null
END
If (@SortOrder = "")
BEGIN
	SET @SortOrder = 0
END
If (@IconPath = "")
BEGIN
	SET @IconPath = null
END
If (@ParentNodeID = "")
BEGIN
	SET @ParentNodeID = null
END

UPDATE InstantKB_Tree SET NodeName = @NodeName, 
ParentNodeID = @ParentNodeID,
SortOrder = @SortOrder, IconPath = @IconPath WHERE NodeID = @NodeID
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminUpdateComment (
@commentID Int,
@comment nvarchar(1000)
)  AS
UPDATE InstantKB_Comments SET comments = @comment WHERE commentID = @commentID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminUpdateLastLogin (
@adminID int
) AS
    
IF EXISTS (SELECT UserID FROM InstantKB_AdminUsers WHERE UserID = @adminID)
BEGIN
	UPDATE InstantKB_AdminUsers SET LastLogin = GetDate() WHERE UserID = @adminID
END
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminUpdateRole (
@RoleID Int,
@RoleDescription nvarchar(255)
)  AS
UPDATE InstantKB_Roles SET RoleDescription = @RoleDescription WHERE RoleID = @RoleID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_AdminUpdateUser (
@UserID Int,
@UserName nvarchar(255),
@Password nvarchar(255)
)  AS
UPDATE InstantKB_Users SET Username = @UserName, [Password] = @Password WHERE UserID = @UserID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_CheckUserRolesAgainstNodeID (
@paramNodeID nvarchar(75),
@userRoles nvarchar(255),
@HasPermission bit OUTPUT
) AS
DECLARE @SQL nvarchar(1000) -- dynamic SQL statement
DECLARE @Delimiter char(1)
DECLARE @separator_position int
DECLARE @array_value varchar(5)
DECLARE @LoopCount int
DECLARE @NodeID varchar(75) -- output parameted from the dynamic SQL used to test existance of result
DECLARE @ParmDefinition nvarchar(500)
SET @LoopCount = 0
SET @Delimiter = ","
SET @NodeID = ""
SELECT @SQL = 'SELECT DISTINCT @NodeID = InstantKB_TreeRoles.NodeID FROM InstantKB_TreeRoles '
SELECT @SQL = @SQL + 'WHERE (InstantKB_TreeRoles.RoleID = 1 '
If (@userRoles <> "")
BEGIN
	SET @userRoles = @userRoles + @Delimiter
	WHILE patindex('%' + @Delimiter  + '%' , @userRoles) <> 0 -- loop until we find no more separators
	BEGIN
		SET @separator_position =  patindex('%' +  @Delimiter + '%' , @userRoles)  -- patindex matches the a pattern against a string
		SET @array_value = left(@userRoles, @separator_position - 1) -- array value holds each single keyword
		
		SELECT @SQL = @SQL + ' OR '
	
		-- @array_value holds the value of this element of the array		
		SELECT @SQL = @SQL + 'InstantKB_TreeRoles.RoleID = ' + @array_value 
		
		SET @LoopCount = + 1
		-- This replaces what we just processed with and empty string, pevents infinate loops
		SET @userRoles = stuff(@userRoles, 1, @separator_position, '')
	END
END
SELECT @SQL = @SQL + ') AND (InstantKB_TreeRoles.NodeID = "' + @paramNodeID + '")'
SET @ParmDefinition = N'@NodeID varchar(75) OUTPUT'
EXEC sp_ExecuteSQL @SQL, @ParmDefinition, @NodeID OUTPUT
IF (@NodeID <> "")
BEGIN
	SET @HasPermission = 1
END
ELSE
BEGIN
	SET @HasPermission = 0
END
RETURN

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetAdminLastLogin (
@AdminID int,
@LastLogin smalldatetime output
) AS
    
SET @LastLogin = (SELECT LastLogin FROM InstantKB_AdminUsers WHERE UserID = @AdminID)
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetAllNodeIDs 
AS
SELECT NodeID FROM InstantKb_Tree 
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetArticle (
@articleID int
) AS
    
SELECT * FROM InstantKB_Articles
WHERE QuestionID = @articleID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetArticleCount (
@nodeID nvarchar(255),
@intArticleCount int output
) AS
SET @intArticleCount = (SELECT COUNT(InstantKB_ArticlesTreeNodes.NodeID) AS Expr1 
			FROM InstantKB_Articles 
			LEFT JOIN InstantKB_ArticlesTreeNodes ON InstantKB_Articles.QuestionID = InstantKB_ArticlesTreeNodes.QuestionID 
			WHERE NodeID = @NodeID AND PublishToWeb = 1)
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetArticleRating (
@articleID int
) AS
    
SELECT rating FROM InstantKB_Ratings
WHERE QuestionID = @articleID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetArticleText (
@Node NVarChar(255)
)
AS

	SELECT InstantKB_Articles.AnswerText, InstantKB_Articles.QuestionID FROM  InstantKB_ArticlesTreeNodes INNER JOIN
	InstantKB_Articles ON InstantKB_ArticlesTreeNodes.QuestionID = InstantKB_Articles.QuestionID
	WHERE InstantKB_ArticlesTreeNodes.NodeID = @Node


return
GO


SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetArticleTitle (
@articleID int,
@articleTitle nvarchar(3000) output
) AS
    
SET @articleTitle = (SELECT QuestionText FROM InstantKB_Articles WHERE QuestionID = @articleID)
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

-- Stored procedure returns the categories that an article is assigned to
CREATE PROCEDURE ikb_sp_GetArticleTreeNodes @ArticleID Int AS
	SELECT InstantKB_Tree.NodeID, InstantKB_Tree.NodeName, InstantKB_Tree.ParentNodeID, InstantKB_Tree.SortOrder, InstantKB_Tree.IconPath
	FROM InstantKB_ArticlesTreeNodes
	LEFT JOIN InstantKB_Tree ON InstantKB_ArticlesTreeNodes.NodeID = InstantKB_Tree.NodeID
	WHERE InstantKB_ArticlesTreeNodes.QuestionID = @ArticleID;

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetArticles (
@nodeID nvarchar(255)
) AS
IF (@nodeID = 'Suggested') -- if nodeID is suggested return all articles for answer questions feature within the administration pages
BEGIN
	SELECT QuestionText, LastModified, QuestionID, DateStamp, ArticleType FROM InstantKB_Articles WHERE SuggestedFlag =1 ORDER BY LastModified DESC, DateStamp DESC, QuestionText ASC
END
ELSE -- else this stored proc is being used to grab only the knowledge base articles with a publishtoweb status for the main default.aspx page
BEGIN
	SELECT DISTINCT QuestionText, LastModified, InstantKB_Articles.QuestionID, ArticleType, LastModified, DateStamp
	FROM InstantKB_Articles LEFT JOIN InstantKB_ArticlesTreeNodes ON InstantKB_Articles.QuestionID = InstantKB_ArticlesTreeNodes.QuestionID
	WHERE InstantKB_ArticlesTreeNodes.NodeID = @NodeID AND PublishToWeb = 1
	ORDER BY LastModified DESC, DateStamp DESC, QuestionText ASC
END 
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO




CREATE Procedure ikb_sp_GetRSS (
@NodeID nvarchar(255),
@userRoles VarChar(75)
) AS
DECLARE @SQL varchar(8000)
DECLARE @Delimiter char(1)
DECLARE @separator_position int -- This is used to locate each separator character
DECLARE @array_value varchar(1000) -- this holds each array value as it is returned
DECLARE @LoopCount int -- loop counter
SET @LoopCount = 0 -- used to count loops when ANY type search is selected
SET @Delimiter = ','
IF (@NodeID = '')
BEGIN
	SELECT @SQL = 'SELECT DISTINCT TOP 50 '
END
ELSE
BEGIN
	SELECT @SQL = 'SELECT DISTINCT '
END
SELECT @SQL = @SQL + 'QuestionText, CAST(AnswerText AS nVarChar(2000)) As AnswerText, InstantKB_Articles.QuestionID, 
InstantKB_Articles.DateStamp, ArticleType, NoOfViews, 
InstantKB_Articles.LastModified, FullName FROM InstantKB_Articles INNER JOIN
InstantKB_ArticlesTreeNodes ON InstantKB_Articles.QuestionID = InstantKB_ArticlesTreeNodes.QuestionID INNER JOIN
InstantKB_Tree ON InstantKB_ArticlesTreeNodes.NodeID = InstantKB_Tree.NodeID INNER JOIN
InstantKB_TreeRoles ON InstantKB_Tree.NodeID = InstantKB_TreeRoles.NodeID INNER JOIN
InstantKB_AdminUsers ON InstantKB_Articles.AdminID = InstantKB_AdminUsers.UserID
WHERE  (InstantKB_TreeRoles.RoleID = 1 '
If (@userRoles <> '')
BEGIN
	SET @userRoles = @userRoles + @Delimiter
	WHILE patindex('%' + @Delimiter  + '%' , @userRoles) <> 0 -- loop until we find no more separators
	BEGIN
		SET @separator_position =  patindex('%' +  @Delimiter + '%' , @userRoles)  -- patindex matches the a pattern against a string
		SET @array_value = left(@userRoles, @separator_position - 1) -- array value holds each single keyword
		
		SELECT @SQL = @SQL + ' OR '
	
		-- @array_value holds the value of this element of the array		
		SELECT @SQL = @SQL + 'InstantKB_TreeRoles.RoleID = ' + @array_value 
		
		SET @LoopCount = + 1
		-- This replaces what we just processed with and empty string, pevents infinate loops
		SET @userRoles = stuff(@userRoles, 1, @separator_position, '')
	END
END
IF (@NodeID = '')
BEGIN
	SELECT @SQL = @SQL + ') AND (PublishToWeb = 1) ORDER BY InstantKB_Articles.LastModified DESC, InstantKB_Articles.DateStamp DESC, InstantKB_Articles.QuestionText ASC'
END
ELSE
BEGIN
	SELECT @SQL = @SQL + ') AND InstantKB_Tree.NodeID = ''' + @NodeID + ''' ORDER BY InstantKB_Articles.LastModified DESC'
END
Exec (@SQL)
return
GO



SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO


CREATE Procedure ikb_sp_GetAttachments (
@articleID int
) AS
    
SELECT * FROM InstantKB_Attachments
WHERE QuestionID = @articleID ORDER BY DateStamp DESC
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetAuthorEmail (
@adminID int,
@AuthorEmail nvarchar(150) output
) AS
    
SET @AuthorEmail = (SELECT EmailAddress FROM InstantKB_AdminUsers WHERE UserID = @adminID)
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetAuthorName (
@adminID int,
@AuthorName nvarchar(150) output
) AS
    
SET @AuthorName = (SELECT FullName FROM InstantKB_AdminUsers WHERE UserID = @adminID)
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetComments (
@articleID Int
)  AS
IF (@ArticleID = 0) 
BEGIN
	SELECT * FROM InstantKB_Comments ORDER BY DateStamp DESC
END
ELSE
BEGIN
	SELECT * FROM InstantKB_Comments WHERE ArticleID = @ArticleID AND Approved = 1 ORDER BY DateStamp DESC
END 
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetKBStatisics (
@TotalCategories int output,
@TotalArticles int output,
@QWaitingApproval int output,
@TotalComments int output,
@TotalCommentsWaitingApproval int output,
@TotalUsers int output,
@TotalUserRoles int output,
@PopularCategory nvarchar(75) output,
@LeastPopularCategory nvarchar(75) output
) AS
    
SET @TotalCategories = (SELECT Count(NodeID) FROM InstantKB_Tree)
SET @TotalArticles = (SELECT Count(QuestionID) FROM InstantKB_Articles WHERE SuggestedFlag = 0)
SET @QWaitingApproval = (SELECT Count(QuestionID) FROM InstantKB_Articles WHERE SuggestedFlag = 1)
SET @TotalComments =  (SELECT Count(CommentID) FROM InstantKB_Comments WHERE Approved = 1)
SET @TotalCommentsWaitingApproval =  (SELECT Count(CommentID) FROM InstantKB_Comments WHERE Approved = 0)
SET @TotalUsers =  (SELECT Count(UserID) FROM InstantKB_Users)
SET @TotalUserRoles =  (SELECT Count(RoleID) FROM InstantKB_Roles)
SET @PopularCategory = (SELECT TOP 1 InstantKB_Tree.NodeID FROM InstantKB_Articles 
	LEFT JOIN InstantKB_ArticlesTreeNodes ON InstantKB_Articles.QuestionID = InstantKB_ArticlesTreeNodes.QuestionID 
	INNER JOIN InstantKB_Tree ON InstantKB_ArticlesTreeNodes.NodeID = InstantKB_Tree.NodeID 
	GROUP BY InstantKB_Tree.NodeID, InstantKB_Tree.NodeName 
	ORDER BY SUM(InstantKB_Articles.NoOfViews) DESC)
SET @LeastPopularCategory = (SELECT TOP 1 InstantKB_Tree.NodeID FROM InstantKB_Articles 
	LEFT JOIN InstantKB_ArticlesTreeNodes ON InstantKB_Articles.QuestionID = InstantKB_ArticlesTreeNodes.QuestionID 
	INNER JOIN InstantKB_Tree ON InstantKB_ArticlesTreeNodes.NodeID = InstantKB_Tree.NodeID 
	GROUP BY InstantKB_Tree.NodeID, InstantKB_Tree.NodeName 
	ORDER BY SUM(InstantKB_Articles.NoOfViews) ASC)
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetLatestArticles (
@userRoles varchar(75)
)AS
DECLARE @SQL varchar(1000)
DECLARE @Delimiter char(1)
DECLARE @separator_position int -- This is used to locate each separator character
DECLARE @array_value varchar(1000) -- this holds each array value as it is returned
DECLARE @LoopCount int -- loop counter
SET @LoopCount = 0 -- used to count loops when ANY type search is selected
SET @Delimiter = ","
SELECT @SQL = 'SELECT DISTINCT TOP 10 QuestionText, LastModified, InstantKB_Articles.QuestionID, DateStamp, ArticleType, NoOfViews, LastModified 
FROM InstantKB_Articles 
INNER JOIN InstantKB_ArticlesTreeNodes ON InstantKB_Articles.QuestionID = InstantKB_ArticlesTreeNodes.QuestionID 
INNER JOIN InstantKB_Tree ON InstantKB_ArticlesTreeNodes.NodeID = InstantKB_Tree.NodeID 
INNER JOIN InstantKB_TreeRoles ON InstantKB_Tree.NodeID = InstantKB_TreeRoles.NodeID 
WHERE (InstantKB_TreeRoles.RoleID = 1 '
If (@userRoles <> "")
BEGIN
	SET @userRoles = @userRoles + @Delimiter
	WHILE patindex('%' + @Delimiter  + '%' , @userRoles) <> 0 -- loop until we find no more separators
	BEGIN
		SET @separator_position =  patindex('%' +  @Delimiter + '%' , @userRoles)  -- patindex matches the a pattern against a string
		SET @array_value = left(@userRoles, @separator_position - 1) -- array value holds each single keyword
		
		SELECT @SQL = @SQL + ' OR '
	
		-- @array_value holds the value of this element of the array		
		SELECT @SQL = @SQL + 'InstantKB_TreeRoles.RoleID = ' + @array_value 
		
		SET @LoopCount = + 1
		-- This replaces what we just processed with and empty string, pevents infinate loops
		SET @userRoles = stuff(@userRoles, 1, @separator_position, '')
	END
END
SELECT @SQL = @SQL + ') AND (PublishToWeb = 1) ORDER BY InstantKB_Articles.LastModified DESC, InstantKB_Articles.DateStamp DESC, InstantKB_Articles.QuestionText ASC'
Exec (@SQL)
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetNavBarLinkInfo (
@nodeID nvarchar(255)
) AS
    
SELECT * FROM InstantKb_Tree WHERE NodeID = @nodeID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetNodeIDsForArticle (
@articleID int
) AS
    
SELECT NodeID FROM InstantKB_ArticlesTreeNodes WHERE QuestionID = @articleID
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetNodeRoleRelationships (
@NodeID varchar(75)
)AS
SELECT InstantKB_Roles.RoleID, InstantKB_Roles.RoleDescription FROM InstantKB_Tree 
INNER JOIN InstantKB_TreeRoles ON InstantKB_Tree.NodeID = InstantKB_TreeRoles.NodeID 
INNER JOIN InstantKB_Roles ON InstantKB_TreeRoles.RoleID = InstantKB_Roles.RoleID 
WHERE InstantKB_Tree.NodeID = @NodeID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetNodeTitle (
@nodeID nvarchar(255),
@nodeName nvarchar(255) output
) AS
    
SET @nodeName = (SELECT NodeName FROM InstantKb_Tree WHERE NodeID = @nodeID)
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetNodes (
@ParentID nvarchar(75),
@userRoles nvarchar(255)
) AS
DECLARE @SQL varchar(2000)
DECLARE @Delimiter char(1)
DECLARE @separator_position int -- This is used to locate each separator character
DECLARE @array_value varchar(1000) -- this holds each array value as it is returned
DECLARE @LoopCount int -- loop counter
SET @LoopCount = 0 -- used to count loops when ANY type search is selected
SET @Delimiter = ","

SELECT @SQL = 'SELECT DISTINCT InstantKB_Tree.* FROM InstantKB_Tree INNER JOIN InstantKB_TreeRoles ON 
InstantKB_Tree.NodeID = InstantKB_TreeRoles.NodeID INNER JOIN InstantKB_Roles ON 
InstantKB_TreeRoles.RoleID = InstantKB_Roles.RoleID WHERE (InstantKB_TreeRoles.RoleID = 1 '
If (@userRoles <> "")
BEGIN
	SET @userRoles = @userRoles + @Delimiter
	WHILE patindex('%' + @Delimiter  + '%' , @userRoles) <> 0 -- loop until we find no more separators
	BEGIN
		SET @separator_position =  patindex('%' +  @Delimiter + '%' , @userRoles)  -- patindex matches the a pattern against a string
		SET @array_value = left(@userRoles, @separator_position - 1) -- array value holds each single keyword
		
		SELECT @SQL = @SQL + ' OR '
	
		-- @array_value holds the value of this element of the array		
		SELECT @SQL = @SQL + 'InstantKB_TreeRoles.RoleID = ' + @array_value 
		
		SET @LoopCount = + 1
		-- This replaces what we just processed with and empty string, pevents infinate loops
		SET @userRoles = stuff(@userRoles, 1, @separator_position, '')
	END
END

IF (@ParentID = "") 
BEGIN 
	SELECT @SQL = @SQL + ') AND ParentNodeID = null'
END
ELSE
BEGIN 
	SELECT @SQL = @SQL + ') AND ParentNodeID = "' + @ParentID + '"'
END

SELECT @SQL = @SQL + ' ORDER BY InstantKB_Tree.SortOrder DESC, InstantKB_Tree.NodeName'
Exec (@SQL)
return
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetParentNodeIDs (
@currentID varchar(75)
) AS
SELECT NodeID, ParentNodeID FROM InstantKb_Tree WHERE NodeID = @currentID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetPopularArticles (
@userRoles varchar(75)
) AS
DECLARE @SQL varchar(1000)
DECLARE @Delimiter char(1)
DECLARE @separator_position int -- This is used to locate each separator character
DECLARE @array_value varchar(1000) -- this holds each array value as it is returned
DECLARE @LoopCount int -- loop counter
SET @LoopCount = 0 -- used to count loops when ANY type search is selected
SET @Delimiter = ","
SELECT @SQL = 'SELECT DISTINCT TOP 10 QuestionText, LastModified, InstantKB_Articles.QuestionID, DateStamp, ArticleType, NoOfViews, LastModified 
FROM InstantKB_Articles 
INNER JOIN InstantKB_ArticlesTreeNodes ON InstantKB_Articles.QuestionID = InstantKB_ArticlesTreeNodes.QuestionID
INNER JOIN InstantKB_Tree ON InstantKB_ArticlesTreeNodes.NodeID = InstantKB_Tree.NodeID 
INNER JOIN InstantKB_TreeRoles ON InstantKB_Tree.NodeID = InstantKB_TreeRoles.NodeID 
WHERE (InstantKB_TreeRoles.RoleID = 1 '
If (@userRoles <> "")
BEGIN
	SET @userRoles = @userRoles + @Delimiter
	WHILE patindex('%' + @Delimiter  + '%' , @userRoles) <> 0 -- loop until we find no more separators
	BEGIN
		SET @separator_position =  patindex('%' +  @Delimiter + '%' , @userRoles)  -- patindex matches the a pattern against a string
		SET @array_value = left(@userRoles, @separator_position - 1) -- array value holds each single keyword
		
		SELECT @SQL = @SQL + ' OR '
	
		-- @array_value holds the value of this element of the array		
		SELECT @SQL = @SQL + 'InstantKB_TreeRoles.RoleID = ' + @array_value 
		
		SET @LoopCount = + 1
		-- This replaces what we just processed with and empty string, pevents infinate loops
		SET @userRoles = stuff(@userRoles, 1, @separator_position, '')
	END
END
SELECT @SQL = @SQL + ') AND (PublishToWeb = 1) ORDER BY InstantKB_Articles.NoOfViews DESC, InstantKB_Articles.LastModified DESC'
Exec (@SQL)
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetRelatedArticles (
@ArticleID int
) AS

SELECT DISTINCT ArticleType, QuestionText, LastModified, InstantKB_ArticlesRelArticles.RelQuestionID As QuestionID, ArticleType, LastModified, DateStamp
FROM InstantKB_ArticlesRelArticles LEFT JOIN InstantKB_Articles ON InstantKB_ArticlesRelArticles.RelQuestionID = InstantKB_Articles.QuestionID
WHERE InstantKB_ArticlesRelArticles.QuestionID = @ArticleID AND PublishToWeb = 1
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetRoles
AS
SELECT RoleID, RoleDescription FROM InstantKB_Roles
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetUserRoleRelationships (
@UserID int
)AS
SELECT InstantKB_Roles.* FROM InstantKB_Users 
INNER JOIN InstantKB_UserRoles ON InstantKB_Users.UserID = InstantKB_UserRoles.UserID 
INNER JOIN InstantKB_Roles ON InstantKB_UserRoles.RoleID = InstantKB_Roles.RoleID
WHERE InstantKB_Users.UserID = @UserID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER ON 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetUserRoles (
@Username nvarchar(255),
@Password nvarchar(255)
) AS
    
DECLARE @userid int
SELECT @userid = UserID  FROM InstantKB_Users WHERE Username = @UserName AND [Password] = @Password
SELECT  InstantKB_Roles.RoleID FROM InstantKB_UserRoles INNER JOIN InstantKB_Roles ON InstantKB_UserRoles.RoleID = InstantKB_Roles.RoleID WHERE InstantKB_UserRoles.UserID = @userid
RETURN

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_GetUserRolesForDDL
AS
    
SELECT * FROM InstantKB_Roles
RETURN

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_IncrementArticleViews (
@articleID Int
)  AS
UPDATE InstantKB_Articles 
SET NoOfViews = NoOfViews + 1
WHERE QuestionID = @articleID
return

GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_ObtainDatabaseInformation
AS
SET NOCOUNT ON
CREATE TABLE #TBLSize
 (Tblname varchar(80), 
 TblRows int,
 TblReserved varchar(80),
 TblData varchar(80),
 TblIndex_Size varchar(80),
 TblUnused varchar(80))
DECLARE @DBname varchar(80) 
DECLARE @tablename varchar(80) 
SELECT @DBname = DB_NAME(DB_ID())
DECLARE TblName_cursor CURSOR FOR 
SELECT name 
FROM sysobjects
WHERE xtype = 'U'
OPEN TblName_cursor
FETCH NEXT FROM TblName_cursor 
INTO @tablename
WHILE @@FETCH_STATUS = 0
BEGIN
   INSERT INTO #tblSize(Tblname, TblRows, TblReserved, TblData, TblIndex_Size, TblUnused)
   EXEC Sp_SpaceUsed @tablename
      
   -- Get the next author.
   FETCH NEXT FROM TblName_cursor 
   INTO @tablename
END
CLOSE TblName_cursor
DEALLOCATE TblName_cursor
SELECT Tblname, TblRows,TblReserved,TblData,TblIndex_Size,TblUnused FROM #tblSize WHERE Tblname LIKE 'InstantKB_%' ORDER BY Tblname
DROP TABLE #TblSize
return


GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_Search (
@strQuery nvarchar(4000), -- keywords passed into the procedure
@strNodeID nvarchar(75), -- category to search within
@strArticleType nvarchar(100), -- article type to search for
@strUsing nvarchar(50), -- type of search to perform - options are ALL, ANY and EXact
@strSearch nvarchar(50), -- field to search options are full title and article id
@strMaximumAge datetime, -- daterange to search
@strUserRoles nvarchar(255) -- use-roles assocaited with user
) AS

DECLARE @SQL varchar(8000)
DECLARE @separator_position int -- This is used to locate each separator character
DECLARE @array_value varchar(500) -- this holds each array value as it is returned
DECLARE @LoopCount int -- used to count loops when ANY type search is selected
DECLARE @separator varchar(1) -- separator used to split query at spaces when searching for any word
DECLARE @userRoleDelimiter varchar(1)  -- separator used to split user-role ids
DECLARE @LOperator varchar(5)
SET @separator = " " 
SET @LoopCount = 0 

SELECT @SQL = 'SELECT DISTINCT InstantKB_Articles.QuestionID, QuestionText, ArticleType, 
CAST(AnswerText AS NVarChar(1000)) As AnswerText, LastModified, DateStamp FROM InstantKB_Articles INNER JOIN
InstantKB_ArticlesTreeNodes ON InstantKB_Articles.QuestionID = InstantKB_ArticlesTreeNodes.QuestionID INNER JOIN
InstantKB_TreeRoles ON InstantKB_ArticlesTreeNodes.NodeID = InstantKB_TreeRoles.NodeID INNER JOIN
InstantKB_Roles ON InstantKB_TreeRoles.RoleID = InstantKB_Roles.RoleID'
IF (@strUsing = "ex" and @strUsing <> "articleID") -- all keywords search
BEGIN
	IF (@strSearch = "full") -- search all fields
		BEGIN
			SELECT @SQL = @SQL + ' WHERE (QuestionText LIKE ''%' + @strQuery  + '%'' or AnswerText LIKE  ''%' + @strQuery + '%'' or
						    Keywords LIKE  ''%' + @strQuery + '%'' or ArticleType LIKE  ''%' + @strQuery + '%'')' 
		END
	IF (@strSearch = "title") -- search title
		BEGIN
			SELECT @SQL = @SQL + ' WHERE (InstantKB_Articles.QuestionText LIKE ''%' + @strQuery  + '%'' or 
			Keywords LIKE  ''%' + @strQuery + '%'' or ArticleType LIKE  ''%' + @strQuery + '%'')' 
		END
END
IF ((@strUsing = "any" or @strUsing = "all") and @strSearch <> "articleID") -- any word within search string - split words
BEGIN
SELECT @SQL = @SQL + ' WHERE '
	SET @strQuery = @strQuery + @separator
		
	-- Loop through the string searching for separtor characters
	WHILE patindex('%' + @separator + '%' , @strQuery) <> 0 -- loop until we find no more separators
	BEGIN
		If (@strUsing = "any") -- if its not the first loop append OR to SQL statement
		BEGIN
			SET @LOperator = " OR "
		END
		ELSE BEGIN
			SET @LOperator = " AND "
		END
		SET @separator_position =  patindex('%' + @separator + '%' , @strQuery)  -- patindex matches the a pattern against a string
		SET @array_value = left(@strQuery, @separator_position - 1) -- array value holds each single keyword
	
		If (@LoopCount <> 0) -- if its not the first loop append OR to SQL statement
		BEGIN
			SELECT @SQL = @SQL + @LOperator
		END
		-- @array_value holds the value of this element of the array
		
		IF (@strSearch = "full") -- search all fields
		  BEGIN
			SELECT @SQL = @SQL + '(QuestionText LIKE ''%' + @array_value + '%'' OR AnswerText LIKE ''%' + @array_value + '%'' OR 
			ArticleType LIKE  ''%' + @array_value + '%'' OR Keywords LIKE  ''%' + @array_value + '%'')' 
		  END
		IF (@strSearch = "title")
		  BEGIN
			SELECT @SQL = @SQL + '(QuestionText LIKE ''%' + @array_value + '%''  or
			Keywords LIKE  ''%' + @array_value + '%'' OR ArticleType = ''%' + @array_value + '%'')' 
		  END
	
		SET @LoopCount = + 1 -- increment loop count		
		SET @strQuery = stuff(@strQuery, 1, @separator_position, '') -- This replaces what we just processed with and empty string, pevents infinate loops
	END
END
IF (@strSearch = "articleID") -- search for article ID
BEGIN
	If (IsNumeric(@strQuery) = 1) 
	BEGIN
		SELECT @SQL = @SQL + ' WHERE (InstantKB_Articles.QuestionID = ' + @strQuery + ')' -- if article ID search is number search the id field
	END
		ELSE
	BEGIN
		SELECT @SQL = @SQL + ' WHERE (QuestionText = ''%' + @strQuery  + '%'')' -- else do search on title
	END
END
SELECT @SQL = @SQL + ' AND (InstantKB_TreeRoles.RoleID = 1'
If (@strUserRoles <> "")
BEGIN
	
	DECLARE @role_separator_position int -- This is used to locate each separator character
	DECLARE @role_array_value varchar(1000) -- this holds each array value as it is returned
	DECLARE @roleLoopCount int -- used to count loops when ANY type search is selected
	
	SET @roleLoopCount = 0 -- reset loop count
	SET @userRoleDelimiter = "," 
	SET @strUserRoles = @strUserRoles + @userRoleDelimiter
	WHILE patindex('%' + @userRoleDelimiter  + '%' , @strUserRoles) <> 0 -- loop until we find no more separators
	BEGIN
		SET @role_separator_position =  patindex('%' + @userRoleDelimiter + '%' , @strUserRoles)  -- patindex matches the a pattern against a string
		SET @role_array_value = left(@strUserRoles, @role_separator_position - 1) -- array value holds each single keyword
		
		SELECT @SQL = @SQL + ' OR '
	
		-- @array_value holds the value of this element of the array		
		SELECT @SQL = @SQL + 'InstantKB_TreeRoles.RoleID = ' + @role_array_value 
		
		SET @roleLoopCount = + 1
		-- This replaces what we just processed with and empty string, pevents infinate loops
		SET @strUserRoles = stuff(@strUserRoles, 1, @role_separator_position, '')
	END
END
IF Len(@strNodeID) > 0
BEGIN
	SELECT @SQL = @SQL + ') AND NodeID=''' + @strNodeID + ''' '
END
ELSE
BEGIN
	SELECT @SQL = @SQL + ') '
END
IF Len(@strArticleType) > 0
BEGIN
	SELECT @SQL = @SQL + 'AND ArticleType=''' + @strArticleType + ''' '
END
SELECT @SQL = @SQL + 'AND (PublishToWeb = 1 AND DateStamp >= ''' + cast(@strMaximumAge as varchar) + ''')' 

Exec (@SQL)
return
GO


SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_FullTextSearch (
@strQuery nvarchar(4000), -- keywords passed into the procedure
@strNodeID nvarchar(75), -- category to search within
@strArticleType nvarchar(100), -- article type to search for
@strUsing nvarchar(50), -- type of boolean search to perform
@strSearch nvarchar(50), -- field to search options are full title and article id
@strMaximumAge datetime, -- daterange to search
@strUserRoles nvarchar(255), -- use-roles assocaited with user
@strFullTextSearchType nvarchar(50) -- the type of fulltext search to perform
) AS
DECLARE @SQL VarChar(8000)
DECLARE @userRoleDelimiter varchar(1)  -- separator used to split user-role ids
IF (@strSearch <> "articleID") -- we are not searching for an article identity so use a full-text search
BEGIN
	IF (@strSearch = "full")
	BEGIN
	
		SELECT @SQL = 'SELECT DISTINCT Article.RANK, InstantKB_Articles.QuestionID, 
		QuestionText, Keywords, ArticleType, 
		CAST(AnswerText AS nVarChar(1000)) As AnswerText, LastModified FROM InstantKB_Articles INNER JOIN
		InstantKB_ArticlesTreeNodes ON InstantKB_Articles.QuestionID = InstantKB_ArticlesTreeNodes.QuestionID INNER JOIN
		InstantKB_TreeRoles ON InstantKB_ArticlesTreeNodes.NodeID = InstantKB_TreeRoles.NodeID INNER JOIN
		InstantKB_Roles ON InstantKB_TreeRoles.RoleID = InstantKB_Roles.RoleID, '
	
		IF (@strFullTextSearchType = "Contains")
		BEGIN
			SELECT @SQL = @SQL + 'CONTAINSTABLE(InstantKB_Articles, *,''' + @strQuery + ''')'
		END
		ELSE IF (@strFullTextSearchType = "FreeText")
		BEGIN
			SELECT @SQL = @SQL + ' FREETEXTTABLE(InstantKB_Articles, *, ''' + @strQuery + ''')'
		END
	
		SELECT @SQL = @SQL + ' AS Article WHERE (Article.[KEY] =  InstantKB_Articles.QuestionID)'
			
	END
	ELSE -- we only search titles
	BEGIN
		SELECT @SQL = 'SELECT DISTINCT Article.RANK, InstantKB_Articles.QuestionID, 
		QuestionText, Keywords, ArticleType, 
		CAST(AnswerText AS nVarChar(1000)) As AnswerText, LastModified FROM InstantKB_Articles INNER JOIN
		InstantKB_ArticlesTreeNodes ON InstantKB_Articles.QuestionID = InstantKB_ArticlesTreeNodes.QuestionID INNER JOIN
		InstantKB_TreeRoles ON InstantKB_ArticlesTreeNodes.NodeID = InstantKB_TreeRoles.NodeID INNER JOIN
		InstantKB_Roles ON InstantKB_TreeRoles.RoleID = InstantKB_Roles.RoleID, '
	
		IF (@strFullTextSearchType = "Contains")
		BEGIN
			SELECT @SQL = @SQL + 'CONTAINSTABLE(InstantKB_Articles, QuestionText,''' + @strQuery + ''')'
		END
		ELSE IF (@strFullTextSearchType = "FreeText")
		BEGIN
			SELECT @SQL = @SQL + ' FREETEXTTABLE(InstantKB_Articles, QuestionText, ''' + @strQuery + ''')'
		END
	
		SELECT @SQL = @SQL + ' AS Article WHERE (Article.[KEY] =  InstantKB_Articles.QuestionID)'
	
	END
END
ELSE
	BEGIN
		SELECT @SQL = 'SELECT DISTINCT InstantKB_Articles.QuestionID, QuestionText, Keywords, ArticleType, 
		CAST(InstantKB_Articles.AnswerText AS nVarChar(1000)) As AnswerText, LastModified FROM InstantKB_Articles INNER JOIN
		InstantKB_ArticlesTreeNodes ON InstantKB_Articles.QuestionID = InstantKB_ArticlesTreeNodes.QuestionID INNER JOIN
		InstantKB_TreeRoles ON InstantKB_ArticlesTreeNodes.NodeID = InstantKB_TreeRoles.NodeID INNER JOIN
		InstantKB_Roles ON InstantKB_TreeRoles.RoleID = InstantKB_Roles.RoleID WHERE (InstantKB_Articles.QuestionID = ''' + @strQuery + ''')'
	END
SELECT @SQL = @SQL + ' AND (InstantKB_TreeRoles.RoleID = 1'
If (@strUserRoles <> "")
BEGIN
	
	DECLARE @role_separator_position int -- This is used to locate each separator character
	DECLARE @role_array_value varchar(1000) -- this holds each array value as it is returned
	DECLARE @roleLoopCount int -- used to count loops when ANY type search is selected
	
	SET @roleLoopCount = 0 -- reset loop count
	SET @userRoleDelimiter = "," 
	SET @strUserRoles = @strUserRoles + @userRoleDelimiter
	WHILE patindex('%' + @userRoleDelimiter  + '%' , @strUserRoles) <> 0 -- loop until we find no more separators
	BEGIN
		SET @role_separator_position =  patindex('%' + @userRoleDelimiter + '%' , @strUserRoles)  -- patindex matches the a pattern against a string
		SET @role_array_value = left(@strUserRoles, @role_separator_position - 1) -- array value holds each single keyword
		
		SELECT @SQL = @SQL + ' OR '
	
		--@array_value holds the value of this element of the array		
		SELECT @SQL = @SQL + 'InstantKB_TreeRoles.RoleID = ' + @role_array_value 
		
		SET @roleLoopCount = + 1
		--This replaces what we just processed with and empty string, pevents infinate loops
		SET @strUserRoles = stuff(@strUserRoles, 1, @role_separator_position, '')
	END
END
IF (LEN(@strNodeID) > 0)
	BEGIN
		SELECT @SQL = @SQL + ') AND InstantKB_ArticlesTreeNodes.NodeID=''' + @strNodeID + ''' '
	END
ELSE
	BEGIN
		SELECT @SQL = @SQL + ') '
	END
IF (LEN(@strArticleType) > 0)
	BEGIN
		SELECT @SQL = @SQL + 'AND ArticleType=''' + @strArticleType + ''' '
	END
SELECT @SQL = @SQL + 'AND (PublishToWeb = 1 AND DateStamp >= ''' + cast(@strMaximumAge as varchar) + ''' AND Article.Rank > 0)'
IF (@strSearch <> "articleID") -- any word within search string - split words
BEGIN
	SELECT @SQL = @SQL + 'ORDER BY Article.Rank DESC'
END
Exec (@SQL)
return
GO



SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO


SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS OFF 
GO

CREATE Procedure ikb_sp_hasChildren (
@nodeID nvarchar(255),
@userRoles nvarchar(255),
@strResult bit OUTPUT
) AS
DECLARE @SQL nvarchar(1000) -- dynamic SQL statement
DECLARE @Delimiter char(1)
DECLARE @separator_position int
DECLARE @array_value varchar(5)
DECLARE @LoopCount int
DECLARE @ParentNodeID varchar(75) -- output parameted from the dynamic SQL used to test existance of result
DECLARE @ParmDefinition nvarchar(500)
SET @LoopCount = 0
SET @Delimiter = ","
SET @ParentNodeID = ""
SELECT @SQL = 'SELECT DISTINCT @ParentNodeID = InstantKB_Tree.ParentNodeID FROM InstantKB_Tree INNER JOIN 
InstantKB_TreeRoles ON InstantKB_Tree.NodeID = InstantKB_TreeRoles.NodeID INNER JOIN InstantKB_Roles ON
InstantKB_TreeRoles.RoleID = InstantKB_Roles.RoleID WHERE (InstantKB_TreeRoles.RoleID = 1 '
If (@userRoles <> "")
BEGIN
	SET @userRoles = @userRoles + @Delimiter
	WHILE patindex('%' + @Delimiter  + '%' , @userRoles) <> 0 -- loop until we find no more separators
	BEGIN
		SET @separator_position =  patindex('%' +  @Delimiter + '%' , @userRoles)  -- patindex matches the a pattern against a string
		SET @array_value = left(@userRoles, @separator_position - 1) -- array value holds each single keyword
		
		SELECT @SQL = @SQL + ' OR '
	
		-- @array_value holds the value of this element of the array		
		SELECT @SQL = @SQL + 'InstantKB_TreeRoles.RoleID = ' + @array_value 
		
		SET @LoopCount = + 1
		-- This replaces what we just processed with and empty string, pevents infinate loops
		SET @userRoles = stuff(@userRoles, 1, @separator_position, '')
	END
END
SELECT @SQL = @SQL + ') AND (ParentNodeID = "' + @nodeID + '")'
SET @ParmDefinition = N'@ParentNodeID varchar(75) OUTPUT'
EXEC sp_ExecuteSQL @SQL, @ParmDefinition, @ParentNodeID OUTPUT
IF (@ParentNodeID <> "")
BEGIN
	SET @strResult = 1
END
ELSE
BEGIN
	SET @strResult = 0
END
RETURN
GO
SET QUOTED_IDENTIFIER OFF 
GO
SET ANSI_NULLS ON 
GO

